from typing import Optional

from ut_mongo_util import CollectionBaseClass

from scripts.constants.db_constants import DBConstants
from scripts.db.mongo.ilens_configuration.collections import collection_user_role


class UserCollectionKeys:
    KEY_LANGUAGE = "language"
    KEY_NAME = "name"
    KEY_USER_ID = "user_id"
    KEY_PROJECT_ID = "project_id"
    KEY_USERNAME = "username"
    KEY_USER_ROLE = "user_role_name"
    KEY_EMAIL = "email"


class UserRole(CollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(mongo_client, database=DBConstants.ilens_configuration, collection=collection_user_role)
        self.project_id = project_id
        self.key_user_id = UserCollectionKeys.KEY_USER_ID
        self.key_project_id = UserCollectionKeys.KEY_PROJECT_ID

    def find_user(self, user_id):
        user = self.find_one(query={"user_id": user_id})
        if user:
            return dict(user)
        return user

    def find_user_name(self, user_id, project_id: Optional[str]):
        query = {"user_role_id": user_id, "project_id": project_id}
        one_user = self.find_one(filter_dict={"user_role_name": 1, "_id": 0}, query=query)
        if one_user is None:
            return one_user
        return one_user["user_role_name"]

    @staticmethod
    def get_users_list(project_id=None):
        query_json = [
            {
                "$group": {
                    "_id": None,
                    "data": {"$push": {"k": {"$ifNull": ["$user_id", ""]}, "v": {"$ifNull": ["$username", ""]}}},
                }
            },
            {"$replaceRoot": {"newRoot": {"$arrayToObject": "$data"}}},
        ]
        if bool(project_id):
            query_json.insert(0, {"$match": {"project_id": project_id}})
        return query_json

    def users_list_by_aggregate(self, query: list):
        return self.aggregate(pipelines=query)

    def find_user_by_project_id(self, user_id, project_id):
        user = self.find_one(query={self.key_user_id: user_id, self.key_project_id: project_id})
        if user:
            return dict(user)
        return user

    def find_user_role_by_id(self, user_role_id, filter_dict=None):
        return self.find_one(query={"user_role_id": user_role_id}, filter_dict=filter_dict)
