import re
from typing import Any, Dict, List, Optional, Union

from pydantic import BaseModel
from ut_mongo_util import CollectionBaseClass

from scripts.constants.db_constants import DBConstants
from scripts.constants.db_constants import User as UserConstants
from scripts.db.mongo.ilens_configuration.collections import collection_user_list
from scripts.db.mongo.ilens_configuration.collections.user_space import UserSpace


class UserCollectionKeys:
    KEY_LANGUAGE = "language"
    KEY_NAME = "name"
    KEY_USER_ID = "user_id"
    KEY_SPACE_ID = "space_id"
    KEY_USERNAME = "username"
    KEY_USER_ROLE = "userrole"
    KEY_EMAIL = "email"


class UserSchema(BaseModel):
    name: Optional[str] = ""
    space_id: Optional[str] = ""
    username: Optional[str] = ""
    password: Optional[str] = ""
    email: Optional[Any] = None
    phonenumber: Optional[Any] = None
    userrole: Optional[List[str]] = None
    user_type: Optional[str] = ""
    user_id: Optional[str] = ""
    client_id: Optional[str] = ""
    created_by: Optional[str] = ""
    encryption_salt: Optional[Dict] = {}
    product_encrypted: Optional[bool] = False
    language: Optional[str] = ""
    passwordReset: Optional[Dict] = {}
    failed_attempts: Optional[int] = 0
    is_user_locked: Optional[bool] = False
    last_failed_attempt: Optional[str] = ""
    location: Optional[str] = ""
    azure_id: Optional[str] = ""
    expires_on: Optional[str] = ""
    token_azure: Optional[str] = ""
    disable_user: Optional[bool] = False
    created_on: Optional[int] = 0
    updated_by: Optional[str] = ""
    updated_on: Optional[int] = 0
    mfa_enabled: Optional[bool] = False
    mfa_configured: Optional[bool] = False
    secret: Optional[str] = ""
    mfa_enabled_on: Optional[int] = 0
    password_added_on: Optional[int] = 0
    default_project: Optional[str] = ""


class User(CollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(mongo_client, database=DBConstants.ilens_configuration, collection=collection_user_list)
        self.user_space_mongo = UserSpace()
        self.project_id = project_id
        self.key_user_id = UserCollectionKeys.KEY_USER_ID
        self.key_space_id = UserCollectionKeys.KEY_SPACE_ID
        self.key_username = UserCollectionKeys.KEY_USERNAME
        self.key_email = UserCollectionKeys.KEY_EMAIL
        self.project_specific_keys = UserConstants.user_project_keys

    def find_user(self, space_id, user_id=None, username=None, email=None, filter_dict=None):
        query = {}
        if user_id:
            query[self.key_user_id] = user_id
        if username:
            query[self.key_username] = username
        if email:
            query[self.key_email] = re.compile(email, re.IGNORECASE)
            query[self.key_email] = email
        user = self.find_decrypted(query=query, filter_dict=filter_dict)
        if user:
            if space_id != user["space_id"]:
                user_space_record = self.user_space_mongo.fetch_user_project(
                    user_id=user.get(self.key_user_id), space_id=space_id
                )
                if user_space_record:
                    for item in self.project_specific_keys:
                        user[item] = user_space_record.get(item, None)
            return UserSchema(**user)
        return user

    @staticmethod
    def get_users_list(project_id=None):
        query_json = [
            {
                "$group": {
                    "_id": None,
                    "data": {"$push": {"k": {"$ifNull": ["$user_id", ""]}, "v": {"$ifNull": ["$username", ""]}}},
                }
            },
            {"$replaceRoot": {"newRoot": {"$arrayToObject": "$data"}}},
        ]
        if bool(project_id):
            query_json.insert(0, {"$match": {"project_id": project_id}})
        return query_json

    def users_list_by_aggregate(self, query: list):
        return self.aggregate(pipelines=query)

    def find_user_by_project_id(self, user_id, project_id):
        user = self.find_one(query={self.key_user_id: user_id, self.key_project_id: project_id})
        if user:
            return dict(user)
        return user

    def get_all_users(self, filter_dict=None, sort=None, skip=0, limit=None, **query):
        users = self.find(filter_dict=filter_dict, sort=sort, skip=skip, limit=limit, query=query)
        if users:
            return list(users)
        return []

    def find_user_role_for_user_id(self, user_id, project_id):
        query = {"user_id": user_id, "project_id": project_id}
        filter_dict = {"userrole": 1, "_id": 0}
        return self.find_one(query=query, filter_dict=filter_dict)

    def find_base_user(self, user_id=None, username=None, email=None, filter_dict=None):
        query = {}
        if user_id:
            query[self.key_user_id] = user_id
        if username:
            query[self.key_username] = username
        if email:
            query[self.key_email] = re.compile(email, re.IGNORECASE)
        if not (user := self.find_decrypted(query=query, filter_dict=filter_dict)):
            return user
        try:
            return UserSchema(**user)
        except Exception:
            return user

    def find_by_space(
        self,
        projections=None,
        sort=None,
        query_dict=None,
        limit=None,
        skip=0,
        **filters,
    ) -> Union[Any, None]:
        query = {}
        if query_dict:
            query |= query_dict
        if filters:
            query.update(filters)
        records = self.find(query, projections, sort=sort, limit=limit, skip=skip)
        if records:
            records = self.get_decrypted_records(records)
        return list(records) if records else []

    def delete_one_user(self, user_id, space_id):
        return self.delete_one(query={self.key_user_id: user_id, self.key_space_id: space_id})
