from ut_mongo_util import CollectionBaseClass

from scripts.constants.common_constants import LookupKeys
from scripts.constants.db_constants import CollectionNames, DBConstants
from scripts.logging import logger


class Lookups(CollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(
            mongo_client, database=DBConstants.ilens_configuration, collection=CollectionNames.lookup_table
        )
        self.project_id = project_id

    @property
    def key_lookup_id(self):
        return LookupKeys.KEY_ID

    @property
    def key_name(self):
        return LookupKeys.KEY_NAME

    def find_all_lookups(self, **query):
        """
        The following function will give all lookups for the given set of
        search parameters as keyword arguments
        :return:
        """
        all_lookups = self.find(**query)
        if not all_lookups:
            return []
        return list(all_lookups)

    def find_by_id(self, lookup_id, project_id):
        """
        The following function will give one lookup for a given set of
        search parameters as keyword arguments
        :return:
        """
        one_lookup = self.find_one(query={self.key_lookup_id: lookup_id, "project_id": project_id})
        if not one_lookup:
            return {}
        return one_lookup

    def find_one_lookup(self, lookup_name, project_id, filter_dict=None):
        query = {self.key_name: lookup_name, "project_id": project_id}
        record = self.find_one(query=query, filter_dict=filter_dict)
        if not record:
            return {}
        return dict(record)

    def find_one_lookup_name(self, lookup_name, filter_dict=None):
        query = {self.key_name: lookup_name}
        record = self.find_one(query=query, filter_dict=filter_dict)
        if not record:
            return {}
        return dict(record)

    def find_by_param(self, **query):
        """
        The following function will give one lookup for a given set of
        search parameters as keyword arguments
        :return:
        """
        one_lookup = self.find(query)
        if one_lookup:
            return list(one_lookup)
        return []

    def update_one_lookup(self, lookup_name, lookup_id, project_id, data):
        """
        The following function will update one lookup in
        tags collection based on the given query
        """
        query_dict = {self.key_name: lookup_name, self.key_lookup_id: lookup_id, "project_id": project_id}
        return self.update_one(data=data, query=query_dict)

    def insert_one_lookup(self, data):
        return self.insert_one(data)

    def delete_one_lookup(self, lookup_id):
        """
        The following function will delete one lookup in
        tags collection based on the given query
        """
        if lookup_id:
            return self.delete_one(query={self.key_lookup_id: lookup_id})
        else:
            return False

    def find_by_aggregate(self, query):
        record = self.aggregate(query)
        if record:
            return list(record)
        else:
            return []

    def map_lookup_keys(self, lookup_name, project_id):
        query = {self.key_name: lookup_name, "project_id": project_id}
        _record = self.find_one(query=query)
        if not _record:
            return {}
        return {record["lookupdata_id"]: record["lookup_value"] for record in _record["lookup_data"]}

    def find_one_and_update(self, query, data, upsert=True):
        try:
            database_name = self.database
            collection_name = self.collection
            db = self.client[database_name]
            collection = db[collection_name]
            response = collection.update_one(query, data, upsert=upsert)
            return response.modified_count
        except Exception as e:
            logger.exception(e)
            raise e

    def find_one_lookup_as_label_value(self, lookup_name, project_id, filter_dict=None):
        query = {self.key_name: lookup_name, "project_id": project_id}
        record = self.find_one(query=query, filter_dict=filter_dict)
        label_value_list = []
        if record:
            for each_lookup in record.get("lookup_data", []):
                label_value_list.append(
                    {"label": each_lookup.get("lookup_value"), "value": each_lookup.get("lookupdata_id")}
                )
        return label_value_list

    def get_lookup_property_values(self, lookup_name, project_id, property, lookup_id_list, filter_dict=None):
        create_property_dict = {}
        query = {self.key_name: lookup_name, "project_id": project_id}
        record = self.find_one(query=query, filter_dict=filter_dict)
        if record:
            for each_lookup in record.get("lookup_data", []):
                if lookup_id_list and each_lookup["lookupdata_id"] not in lookup_id_list:
                    continue
                for each_property in each_lookup["properties"]:
                    if each_property["key"] == property:
                        create_property_dict[each_lookup["lookupdata_id"]] = each_property["value"]
                        break
        return create_property_dict
