import os
import pathlib
import shutil
import sys
from typing import Annotated, Any, Optional

from dotenv import load_dotenv
from pydantic.functional_validators import BeforeValidator
from pydantic.v1 import BaseSettings, Field, root_validator

load_dotenv()

PROJECT_NAME = "ftdm-catalog"


def options_decoder(v):
    if isinstance(v, str):
        return v.split(",")
    return v


OptionsType = Annotated[Any, BeforeValidator(options_decoder)]


class _Service(BaseSettings):
    MODULE_NAME: str = Field(default="Catalog-Management")
    APP_NAME: str = Field(default="catalog-management")
    HOST: str = Field(default="0.0.0.0")
    PORT: int = Field(default=45561)
    LOG_LEVEL: str = Field(default="INFO")
    ENABLE_FILE_LOG: Optional[Any] = False
    ENABLE_CONSOLE_LOG: Optional[Any] = True
    GLOBAL_CATALOG_SERVICES: str = Field(default="")
    GLOBAL_CATALOG_PROJECT_ID: str = Field(default="")
    GLOBAL_CATALOG_USER: str = Field(default="")
    GLOBAL_CATALOG_TOKEN: str = Field(default="")
    GC_BEARER_TOKEN: str = Field(default="")
    TIMEOUT: int = Field(default=60)
    REFRESH_TOKEN_DURATION: int = Field(default=168)
    COOKIE_MAX_AGE_IN_MINS: int = Field(default=60)

    @root_validator(allow_reuse=True)
    def validate_values(cls, values):
        values["LOG_LEVEL"] = values["LOG_LEVEL"] or "INFO"
        print(f"Logging Level set to: {values['LOG_LEVEL']}")
        return values


class _StoragePaths(BaseSettings):
    MODULE_NAME: str = "catalog-v2"
    BASE_PATH: str
    REPORT_PATH: str = Field(None)

    @root_validator(allow_reuse=True)
    def assign_values(cls, values):
        values["BASE_PATH"] = os.path.join("data", values.get("MODULE_NAME"))
        if not values["BASE_PATH"]:
            print("Error, environment variable BASE_PATH not set")
            sys.exit(1)
        values["REPORT_PATH"] = os.path.join(values.get("BASE_PATH"), "reports")
        return values


class _PathToStorage(BaseSettings):
    BASE_PATH: pathlib.Path = Field(None, env="BASE_PATH")
    MOUNT_DIR: pathlib.Path = Field(None, env="MOUNT_DIR")
    TEMP_PATH: pathlib.Path = Field(None, env="TEMP_PATH")
    MODULE_PATH: Optional[pathlib.Path]
    CAPTCHA: str = Field(default="captcha")
    CAPTCHA_PATH: Optional[pathlib.Path]

    @root_validator(allow_reuse=True)
    def assign_values(cls, values):
        values["LOGS_MODULE_PATH"] = os.path.join(values.get("BASE_PATH"), "logs", values.get("MOUNT_DIR"))
        values["MODULE_PATH"] = os.path.join(values.get("BASE_PATH"), values.get("MOUNT_DIR"))
        values["CAPTCHA_PATH"] = os.path.join(values.get("BASE_PATH"), values.get("CAPTCHA"))
        return values

    @root_validator(allow_reuse=True)
    def validate_values(cls, values):
        if not values["BASE_PATH"]:
            print("Error, environment variable BASE_PATH not set")
            sys.exit(1)
        if not values["MOUNT_DIR"]:
            print("Error, environment variable MOUNT_DIR not set")
            sys.exit(1)
        return values


class _KeyPath(BaseSettings):
    KEYS_PATH: Optional[pathlib.Path] = Field(default="data/keys")
    PUBLIC: Optional[pathlib.Path]
    PRIVATE: Optional[pathlib.Path]

    @root_validator(allow_reuse=True)
    def assign_values(cls, values):
        if not os.path.isfile(os.path.join(values.get("KEYS_PATH"), "public")) or not os.path.isfile(
            os.path.join(values.get("KEYS_PATH"), "private")
        ):
            if not os.path.exists(values.get("KEYS_PATH")):
                os.makedirs(values.get("KEYS_PATH"))
            shutil.copy(os.path.join("assets", "keys", "public"), os.path.join(values.get("KEYS_PATH"), "public"))
            shutil.copy(os.path.join("assets", "keys", "private"), os.path.join(values.get("KEYS_PATH"), "private"))
        values["PUBLIC"] = os.path.join(values.get("KEYS_PATH"), "public")
        values["PRIVATE"] = os.path.join(values.get("KEYS_PATH"), "private")
        return values


class _MQTTConf(BaseSettings):
    MQTT_HOST: str
    MQTT_PORT: int
    MQTT_USERNAME: str
    MQTT_PASSWORD: str
    PUBLISH_BASE_TOPIC: str = "ilens/notifications"


class _KeyCloakConf(BaseSettings):
    keycloak_url: Optional[str]
    keycloak_realm: Optional[str]
    keycloak_admin_user: Optional[str]
    keycloak_admin_password: Optional[str]


class _UnifyTwinEmailConf:
    UT_EMAIL_SERVICE: Optional[str] = Field(default="UT_EMAIL_SERVICE")
    URL: Optional[str]
    DIGEST_USER: Optional[str] = Field(default="DIGEST_USER")
    DIGEST_PW: Optional[str] = Field(default="DIGEST_PW")

    @root_validator(allow_reuse=True)
    def assign_values(cls, values):
        values["URL"] = f"{values.get('UT_EMAIL_SERVICE')}/api/v1/eim/email/send"
        return values


class _RedisConf(BaseSettings):
    REDIS_URI: str
    REDIS_LOGIN_DB: int = Field(default=9)
    REDIS_PROJECT_DB: int = Field(default=18)
    REDIS_USER_ROLE_DB: int = Field(default=21)
    REDIS_LICENSE_CHECK_DB: int = Field(default=70)


class _Security(BaseSettings):
    SECURE_COOKIE: bool = Field(default=True)
    VERIFY_SIGNATURE: bool = Field(default=False)
    PROTECTED_HOSTS: list = os.environ.get("PROTECTED_HOSTS", default="").split(",")
    PASSWORD_DECRYPTION_KEY: str = "QVY1bWdMQ0Zxc"
    DISABLE_ENC: bool | str = Field(default=False) in ["True", "true", True]
    VALIDATE_LIMIT: bool | str = Field(default=False) in ["True", "true", True]


Service = _Service()
StoragePaths = _StoragePaths()
PathToStorage = _PathToStorage()
KeyCloakConf = _KeyCloakConf()
KeyPath = _KeyPath()
MQTTConf = _MQTTConf()
Security = _Security()
RedisConf = _RedisConf()
UnifyTwinEmailConf = _UnifyTwinEmailConf()

__all__ = [
    "PROJECT_NAME",
    "Service",
    "StoragePaths",
    "PathToStorage",
    "KeyPath",
    "MQTTConf",
    "KeyCloakConf",
    "Security",
    "RedisConf",
    "UnifyTwinEmailConf",
]
