import logging
from pymongo import MongoClient
from dotenv import load_dotenv
import time
import os
load_dotenv()
import sys
import httpx
from app_clone_script_helper import CloneScriptHelper

root_path = os.getcwd()
print(f'ROOT PATH ---> {root_path}')
sys.path.append(root_path)

MONGO_URI = os.environ.get("MONGO_URI")
BASE_URI = os.environ.get("BASE_URI")
LOGIN_TOKEN = os.environ.get("LOGIN_TOKEN")
USER_ID = os.environ.get("USER_ID")
PROJECT_ID = os.environ.get("PROJECT_ID")
APP_NAME = os.environ.get("APP_NAME")
PREFIX = os.environ.get("PREFIX")
BASE_PATH = os.environ.get("BASE_PATH")
MOUNT_DIR = os.environ.get("MOUNT_DIR")

client = MongoClient(MONGO_URI)
logging.basicConfig(level=logging.DEBUG,  # Set the minimum logging level
                    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')


dashboard_image_uri = f"{BASE_URI}visual4.0/dashboard/snap_dashboard"
hierarchy_service_url = f"{BASE_URI}/hry/hry/fetch_accessible_data"

cookie = {"login-token": LOGIN_TOKEN, "userId": USER_ID, "projectId": PROJECT_ID}


class Database:
    def __init__(self, prefix=None):
        self.assistant = client[f"{prefix}__ilens_assistant"] if prefix else client["ilens_assistant"]
        self.configuration = client[f"{prefix}__ilens_configuration"] if prefix else client["ilens_configuration"]
        self.widget = client[f"{prefix}__ilens_widget"] if prefix else client["ilens_widget"]


class Connection(Database):
    def __init__(self, prefix=None):
        super().__init__(prefix)
        self.dashboard = self.widget["dashboard"]
        self.customer_apps = self.configuration["customer_apps"]
        self.lookup_table = self.configuration["lookup_table"]
        self.category_apps = self.configuration["category_apps"]
        self.dashboard_category = self.widget["category"]
        self.widget_col = self.widget["widget"]
        self.user_conn = self.configuration["user"]
        self.hierarchy_details = self.configuration["hierarchy_details"]
        self.unique_id = self.configuration["unique_id"]
        self.units = self.configuration["units"]
        self.units_group = self.configuration["unit_group"]
        self.tags = self.configuration["tags"]
        self.tag_groups = self.configuration["tag_groups"]
        self.tag_category = self.configuration["tag_category"]


def get_next_id(param, prefix=None):
    connection = Connection(prefix=prefix)
    next_id_doc = connection.unique_id.find_one({"key": param})
    if not next_id_doc:
        insert_dict = {"key": param, "id": 100}
        connection.unique_id.insert_one(insert_dict)
        return insert_dict["id"]
    else:
        query = {"key": param}
        count_value = int(next_id_doc["id"]) + 1
        new_values = {"id": count_value}
        connection.unique_id.update_one(query, {"$set": new_values})
        return int(new_values["id"])


def push_category(folder_path, folder_name, project_id, prefix=None):
    connection = Connection(prefix=prefix)
    file_path = os.path.join(folder_path, f"{folder_name}.json")
    category_data = CloneScriptHelper().read_json_file(file_path)
    for data in category_data:
        data["project_id"] = project_id
        query = {"dashboard_category_id": data.get("dashboard_category_id")}
        connection.dashboard_category.update_one(query, {"$set": data}, upsert=True)


def push_app_category(folder_path, folder_name, user_details, project_id, prefix=None):
    connection = Connection(prefix=prefix)
    file_path = os.path.join(folder_path, f"{folder_name}.json")
    app_category_data = CloneScriptHelper().read_json_file(file_path)
    zip_app_category_data = {}
    for app_category in app_category_data:
        zip_app_category_data[app_category.get('name')] = app_category.get('app_category_id')
    existed_app_category_data = list(
                connection.category_apps.find(
                    {"name": {"$in": list(zip_app_category_data.keys())}, "project_id": project_id}, {"_id": 0}
                )
            )
    existed_app_category_mapping_data = {}
    for existed_app_category in existed_app_category_data:
        existed_app_category_mapping_data[existed_app_category.get('name')] = existed_app_category
    app_category_mapping = {}
    for app_category in app_category_data:
        old_app_category_id = app_category.get("app_category_id")
        if app_category.get('name') in existed_app_category_mapping_data:
            app_category_data = existed_app_category_mapping_data.get(app_category.get("name"))
            app_category_data["project_id"] = project_id
            app_category_data['created_at'] = time.time()
            app_category_data['created_by'] = user_details.get('user_id', '')
            query = {"app_category_id": app_category_data.get("app_category_id")}
            connection.category_apps.update_one(query, {"$set": app_category_data}, upsert=True)
            app_category_mapping[old_app_category_id] = app_category_data.get("app_category_id")
            continue
        app_category["project_id"] = project_id
        app_category['created_at'] = time.time()
        app_category['created_by'] = user_details.get('user_id', '')
        app_category["app_category_id"] = "app_category_" + str(get_next_id("app_category", prefix))
        query = {"name": app_category.get("name")}
        connection.category_apps.update_one(query, {"$set": app_category}, upsert=True)
        app_category_mapping[old_app_category_id] = app_category.get("app_category_id")
    return app_category_mapping


def push_lookups(folder_path, folder_name, prefix=None):
    connection = Connection(prefix=prefix)
    file_path = os.path.join(folder_path, f"{folder_name}.json")
    category_data = CloneScriptHelper().read_json_file(file_path)
    for data in category_data:
        query = {"lookup_id": data.get("lookup_id")}
        connection.lookup_table.update_one(query, {"$set": data}, upsert=True)


def fetch_hierarchy_details(project_id, node_id):
    payload = {"project_id": project_id, "node_id": node_id, "fetch_path_names": True, "fetch_child": None}
    with httpx.Client() as client:
        response = client.post(
            url=hierarchy_service_url,
            timeout=20,
            json=payload,
            cookies=cookie,
            headers=cookie,
        )
    if response.status_code == 200:
        result = response.json()
        if not len(result["data"]):
            return {}
        data = result["data"][0]
        full_path = data.get("full_path", " ")
        return {
                "id": full_path,
                "name": data.get("full_name", " "),
                "node_id": data.get("node_id", " "),
                "site_id": full_path.split("$")[0],
                "type": data.get("type", " "),
                "lat": data.get("info").get("latitude", " "),
                "long": data.get("info").get("longitude", " "),
                "label": data.get("full_name", " "),
                "value": full_path,
                "disabled": False,
                "checked": True,
            }


def process_x_axis(x_axis, hierarchy_list, parameter_id_mapping, widget_type):
    if widget_type != "dynamic":
        if type(x_axis) != str and len(x_axis.get("aggregations")):
            for hierarchy in hierarchy_list:
                for aggregation in x_axis.get("aggregations"):
                    parameters = [f"{hierarchy.get('id')}${parameter_id_mapping.get(tag)}" for tag in
                                  aggregation.get('parameters')]
                    aggregation['parameters'] = parameters
    else:
        if type(x_axis) != str and len(x_axis.get("aggregations")):
            for aggregation in x_axis.get("aggregations"):
                parameters = [parameter_id_mapping.get(tag) for tag in
                              aggregation.get('parameters')]
                aggregation['parameters'] = parameters


def process_axis(y_axis, hierarchy_list, parameter_id_mapping, unit_id_mapping, widget_type):
    if widget_type != "dynamic":
        for hierarchy in hierarchy_list:
            for axis_val in y_axis:
                if type(axis_val) != dict or not axis_val.get("tag"):
                    return
                if axis_val.get("unitInfo"):
                    axis_val.get("unitInfo").update({"value": unit_id_mapping.get(axis_val.get("unitInfo", {}).get("value", ""))})
                for tag_type in ["tag", "forecasted_tag"]:
                    tag = axis_val.get(tag_type)
                    if not tag:
                        continue
                    tag.update(value=f"{hierarchy.get('id')}${parameter_id_mapping.get(tag.get('value'))}")
                    tag.get("unit").update({"value": unit_id_mapping.get(tag.get("unit", {}).get("value"))})
    else:
        for axis_val in y_axis:
            if type(axis_val) != dict or not axis_val.get("tag"):
                return
            if axis_val.get("unitInfo"):
                axis_val.get("unitInfo").update(
                    {"value": unit_id_mapping.get(axis_val.get("unitInfo", {}).get("value", ""))})
            for tag_type in ["tag", "forecasted_tag"]:
                tag = axis_val.get(tag_type)
                if not tag:
                    continue
                tag.update(value=parameter_id_mapping.get(tag.get('value')))
                tag.get("unit").update({"value": unit_id_mapping.get(tag.get("unit", {}).get("value"))})


def manipulate_chart_options_widget(hierarchy_list, y_axis, x_axis, filter_parameters, widget_type, unit_id_mapping, parameter_id_mapping):
    process_axis(y_axis, hierarchy_list, parameter_id_mapping, unit_id_mapping, widget_type)
    process_x_axis(x_axis, hierarchy_list, parameter_id_mapping, widget_type)
    if len(filter_parameters):
        process_axis(filter_parameters, hierarchy_list, parameter_id_mapping, unit_id_mapping, widget_type)


def manipulate_widget_data(widget_data, project_id, hierarchy_list, unit_id_mapping, parameter_id_mapping):
    widget_data["project_id"] = project_id
    if "filterList" in widget_data["widget_data"]["cData"]["chartOptions"]["filter"] and "filtersData" in widget_data["widget_data"]["cData"]["chartOptions"]["filter"]:
        for filter_data in widget_data["widget_data"]["cData"]["chartOptions"]["filter"]["filterList"]:
            filter_data['value'] = hierarchy_list
        for filter_data in widget_data["widget_data"]["cData"]["chartOptions"]["filter"]["filtersData"]:
            filter_data['value'] = hierarchy_list
    y_axis = widget_data["widget_data"]["cData"]["chartOptions"].get("yaxis", " ")
    x_axis = widget_data["widget_data"]["cData"]["chartOptions"].get("xaxis", " ")
    filter_parameters = widget_data["widget_data"]["cData"]["chartOptions"].get("filterParameters", " ")
    widget_type = widget_data["widget_data"]["cData"]["chartOptions"].get("widget_type")
    manipulate_chart_options_widget(hierarchy_list, y_axis, x_axis, filter_parameters, widget_type, unit_id_mapping, parameter_id_mapping)
    widget_data["widget_data"]["cData"]["chartOptions"]['yaxis'] = y_axis
    widget_data["widget_data"]["cData"]["chartOptions"]['xaxis'] = x_axis
    widget_data["widget_data"]["cData"]["chartOptions"]['filterParameters'] = filter_parameters


def push_widget(folder_path, folder_name, project_id, unit_id_mapping, parameter_id_mapping, prefix=None):
    connection = Connection(prefix=prefix)
    file_path = os.path.join(folder_path, f"{folder_name}.json")
    widget_data = CloneScriptHelper().read_json_file(file_path)
    hierarchy_details = connection.hierarchy_details.find({}, {"_id": 0})
    hierarchy_list = []
    for hierarchy in hierarchy_details:
        hierarchy_data = fetch_hierarchy_details(project_id, hierarchy.get("node_id"))
        hierarchy_list.append(hierarchy_data)
    for data in widget_data:
        manipulate_widget_data(data, project_id, hierarchy_list, unit_id_mapping, parameter_id_mapping)
        query = {"widget_id": data.get("widget_id")}
        connection.widget_col.update_one(query, {"$set": data}, upsert=True)


def manipulate_dashboard_data(dashboard_data, user_details, project_id):
    dashboard_data["user_details"] = {
        "username": user_details.get("username"),
        "email": user_details.get("email"),
        "name": user_details.get("name"),
        "user_id": user_details.get("user_id"),
        "access_group_ids": []}
    dashboard_data["owner"] = user_details.get("user_id")
    dashboard_data["project_id"] = project_id
    dashboard_data["shared_group"] = []
    dashboard_data["shared_user"] = []
    dashboard_data["sharing_info"]["users"] = []
    dashboard_data["sharing_info"]["userGroups"] = []


def push_dashboard(folder_path, folder_name, user_details, project_id, prefix=None):
    dashboard_image_push(folder_path, prefix)
    connection = Connection(prefix=prefix)
    file_path = os.path.join(folder_path, f"{folder_name}.json")
    dashboard_data = CloneScriptHelper().read_json_file(file_path)
    for data in dashboard_data:
        manipulate_dashboard_data(data, user_details, project_id)
        query = {"dashboard_id": data.get("dashboard_id")}
        connection.dashboard.update_one(query, {"$set": data}, upsert=True)


def dashboard_image_push(folder_path, prefix=None):
    image_list = CloneScriptHelper().list_images_in_folder(folder_path)
    if not image_list:
        return
    for image in image_list:
        image_path = os.path.join(folder_path, image)
        with open(image_path, 'rb') as image_file:
            dashboard_id = image.split(".")[0]
            form_data = {'image': (image_file.name, image_file, 'image'), "project_id": prefix, "dashboard_id": dashboard_id}
            data = {"dashboard_id": dashboard_id, "project_id": prefix}
            request = httpx.Request('POST', dashboard_image_uri, files=form_data, headers=cookie, cookies=cookie, data=data)
            with httpx.Client() as client:
                response = client.send(request)
            if response.status_code == 200:
                print("Image successfully sent.")
            else:
                print(f"Failed to send image. Status code: {response.status_code}")


def manipulate_app_data(each_children_obj, required_type, user_id, project_id):
    if not each_children_obj.get("children", []):
        view_item = each_children_obj.get("view_item", {})
        if view_item:
            each_children_obj['view_item']['project_id'] = project_id
            each_children_obj['view_item']['owner'] = user_id
        hierarchy_stepper = each_children_obj.get("hierarchyStepper", [])
        if hierarchy_stepper:
            for each_hierarchy_stepper in hierarchy_stepper:
                manipulate_app_data(each_hierarchy_stepper, required_type, user_id, project_id)
    else:
        for each_child in each_children_obj.get("children", []):
            manipulate_app_data(each_child, required_type, user_id, project_id)


def push_unit_group(folder_path, folder_name, project_id, prefix=None):
    connection = Connection(prefix=prefix)
    unit_group_file_path = os.path.join(folder_path, f"{folder_name}.json")
    unit_group_data = CloneScriptHelper().read_json_file(unit_group_file_path)
    zip_unit_group_data = {}
    for unit_group in unit_group_data:
        zip_unit_group_data[unit_group.get('group_name')] = unit_group.get('id')
    existed_unit_group_data = list(
                connection.units_group.find(
                    {"group_name": {"$in": list(zip_unit_group_data.keys())}, "project_id": project_id}, {"_id": 0}
                )
            )
    existed_unit_group_mapping_data = {}
    for existed_unit_group in existed_unit_group_data:
        existed_unit_group_mapping_data[existed_unit_group.get('group_name')] = existed_unit_group

    unit_group_mapping = {}
    for unit_group in unit_group_data:
        old_unit_group_id = unit_group.get("id")
        if unit_group.get('group_name') in existed_unit_group_mapping_data:
            unit_group_data = existed_unit_group_mapping_data.get(unit_group.get("group_name"))
            unit_group_data["project_id"] = project_id
            query = {"id": unit_group_data.get("id")}
            connection.units_group.update_one(query, {"$set": unit_group_data}, upsert=True)
            unit_group_mapping[old_unit_group_id] = unit_group_data.get("id")
            continue
        unit_group["project_id"] = project_id
        unit_group["id"] = "group_" + str(get_next_id("group", prefix))
        query = {"group_name": unit_group.get("group_name")}
        connection.units_group.update_one(query, {"$set": unit_group}, upsert=True)
        unit_group_mapping[old_unit_group_id] = unit_group.get("id")
    return unit_group_mapping


def push_units(folder_path, collection_data, project_id, prefix=None):
    connection = Connection(prefix=prefix)
    unit_file_path = os.path.join(folder_path, collection_data.get("units"), f"{collection_data.get('units')}.json")
    unit_data = CloneScriptHelper().read_json_file(unit_file_path)
    unit_group_folder_path = os.path.join(folder_path, collection_data.get("unit_group"))
    unit_group_mapping = push_unit_group(unit_group_folder_path, collection_data.get("unit_group"), project_id, prefix)
    zip_unit_data = {}
    for unit in unit_data:
        zip_unit_data[unit.get('name')] = unit.get('id')
    existed_unit_data = list(
                connection.units.find(
                    {"name": {"$in": list(zip_unit_data.keys())}, "project_id": project_id}, {"_id": 0}
                )
            )
    existed_unit_mapping_data = {}
    for existed_unit in existed_unit_data:
        existed_unit_mapping_data[existed_unit.get('name')] = existed_unit
    unit_id_mapping = {}
    for unit in unit_data:
        old_unit_id = unit.get("id")
        if unit.get('name') in existed_unit_mapping_data:
            unit_data = existed_unit_mapping_data.get(unit.get("name"))
            query = {"id": unit_data.get("id")}
            connection.units.update_one(query, {"$set": unit_data}, upsert=True)
            unit_id_mapping[old_unit_id] = unit_data.get("id")
            continue
        unit["project_id"] = project_id
        unit["id"] = "unit_" + str(get_next_id("unit", prefix))
        unit['unit_group_id'] = unit_group_mapping.get(unit.get("unit_group_id"), unit.get("unit_group_id"))
        query = {"name": unit.get("name")}
        connection.units.update_one(query, {"$set": unit}, upsert=True)
        unit_id_mapping[old_unit_id] = unit.get("id")
    return unit_id_mapping


def push_tag_category(folder_path, folder_name, prefix=None):
    connection = Connection(prefix=prefix)
    tag_category_file_path = os.path.join(folder_path, f"{folder_name}.json")
    tag_category_data = CloneScriptHelper().read_json_file(tag_category_file_path)
    zip_tag_category_data = {}
    for tag_category in tag_category_data:
        zip_tag_category_data[tag_category.get('tag_category_name')] = tag_category.get('tag_category_id')
    existed_tag_category_data = list(
                connection.tag_category.find(
                    {"tag_category_name": {"$in": list(zip_tag_category_data.keys())}}, {"_id": 0}
                )
            )
    existed_tag_category_mapping_data = {}
    for existed_tag_category in existed_tag_category_data:
        existed_tag_category_mapping_data[existed_tag_category.get('tag_category_name')] = existed_tag_category
    tag_category_mapping = {}
    for tag_category in tag_category_data:
        old_tag_category_id = tag_category.get("tag_category_id")
        if tag_category.get('tag_category_name') in existed_tag_category_mapping_data:
            tag_category_data = existed_tag_category_mapping_data.get(tag_category.get("tag_category_name"))
            query = {"tag_category_id": tag_category_data.get("tag_category_id")}
            connection.tag_category.update_one(query, {"$set": tag_category_data}, upsert=True)
            tag_category_mapping[old_tag_category_id] = tag_category_data.get("tag_category_id")
            continue
        tag_category["tag_category_id"] = "tag_category_" + str(get_next_id("tag_category", prefix))
        query = {"tag_category_name": tag_category.get("tag_category_name")}
        connection.units_group.update_one(query, {"$set": tag_category}, upsert=True)
        tag_category_mapping[old_tag_category_id] = tag_category.get("tag_category_id")
    return tag_category_mapping


def push_tag_group(folder_path, folder_name, prefix=None):
    connection = Connection(prefix=prefix)
    tag_group_file_path = os.path.join(folder_path, f"{folder_name}.json")
    tag_group_data = CloneScriptHelper().read_json_file(tag_group_file_path)
    zip_tag_group_data = {}
    for tag_group in tag_group_data:
        zip_tag_group_data[tag_group.get('tag_group_name')] = tag_group.get('tag_group_id')
    existed_tag_group_data = list(
                connection.tag_groups.find(
                    {"tag_group_name": {"$in": list(zip_tag_group_data.keys())}}, {"_id": 0}
                )
            )
    existed_tag_group_mapping_data = {}
    for existed_tag_group in existed_tag_group_data:
        existed_tag_group_mapping_data[existed_tag_group.get('tag_group_name')] = existed_tag_group
    tag_group_mapping = {}
    for tag_group in tag_group_data:
        old_tag_group_id = tag_group.get("tag_group_id")
        if tag_group.get('tag_group_name') in existed_tag_group_mapping_data:
            tag_group_data = existed_tag_group_mapping_data.get(tag_group.get("tag_group_name"))
            query = {"tag_group_id": tag_group_data.get("tag_group_id")}
            connection.tag_groups.update_one(query, {"$set": tag_group_data}, upsert=True)
            tag_group_mapping[old_tag_group_id] = tag_group_data.get("tag_group_id")
            continue
        tag_group["tag_group_id"] = "tag_group_" + str(get_next_id("tag_group", prefix))
        query = {"tag_group_name": tag_group.get("tag_group_name")}
        connection.units_group.update_one(query, {"$set": tag_group}, upsert=True)
        tag_group_mapping[old_tag_group_id] = tag_group.get("tag_group_id")
    return tag_group_mapping


def push_parameters(folder_path, collection_data, unit_id_mapping, project_id, prefix=None):
    connection = Connection(prefix=prefix)
    tag_file_path = os.path.join(folder_path, collection_data.get("tags"), f"{collection_data.get('tags')}.json")
    tag_data = CloneScriptHelper().read_json_file(tag_file_path)
    tag_group_folder_path = os.path.join(folder_path, collection_data.get("tag_groups"))
    tag_group_mapping = push_tag_group(tag_group_folder_path, collection_data.get("tag_groups"), prefix)
    tag_category_folder_path = os.path.join(folder_path, collection_data.get("tag_category"))
    tag_category_mapping = push_tag_category(tag_category_folder_path, collection_data.get("tag_category"), prefix)
    zip_tag_data = {}
    for tag in tag_data:
        zip_tag_data[tag.get('tag_name')] = tag.get('id')
    existed_tag_data = list(
                connection.tags.find(
                    {"tag_name": {"$in": list(zip_tag_data.keys())}, "project_id": project_id}, {"_id": 0}
                )
            )
    existed_tag_mapping_data = {}
    for existed_tag in existed_tag_data:
        existed_tag_mapping_data[existed_tag.get('tag_name')] = existed_tag
    parameter_id_mapping = {}
    for tag in tag_data:
        old_tag_id = tag.get("tag_id")
        if tag['tag_name'] in existed_tag_mapping_data:
            tag_data = existed_tag_mapping_data.get(tag.get("tag_name"))
            tag_data["project_id"] = project_id
            tag_data["tag_group_id"] = tag_group_mapping.get(tag_data.get("tag_group_id"), tag_data.get("tag_group_id"))
            tag_data["tag_category_id"] = tag_category_mapping.get(tag_data.get("tag_category_id"), tag_data.get("tag_category_id"))
            tag_data['unit'] = unit_id_mapping.get(tag_data.get("unit"), tag_data.get("unit"))
            query = {"tag_id": tag_data.get("tag_id")}
            connection.tags.update_one(query, {"$set": tag_data}, upsert=True)
            parameter_id_mapping[old_tag_id] = tag_data.get("tag_id")
            continue
        tag["project_id"] = project_id
        tag["tag_group_id"] = tag_group_mapping.get(tag.get("tag_group_id"), "")
        tag["tag_category_id"] = tag_category_mapping.get(tag.get("tag_category_id"), "")
        tag['unit'] = unit_id_mapping.get(tag.get("unit"), "")
        tag["id"] = tag["tag_id"] = "tag_" + str(get_next_id("tag", prefix))
        query = {"tag_name": tag.get("tag_name")}
        connection.tags.update_one(query, {"$set": tag}, upsert=True)
        parameter_id_mapping[old_tag_id] = tag.get("tag_id")
    return parameter_id_mapping


def get_new_tag_and_unit_ids(folder_path, collection_data, project_id, prefix=None):
    unit_id_mapping = push_units(folder_path, collection_data, project_id, prefix)
    parameter_id_mapping = push_parameters(folder_path, collection_data, unit_id_mapping, project_id, prefix)
    return unit_id_mapping, parameter_id_mapping


def push_app(folder_path, collection_data, user_id, user_details, project_id, prefix=None):
    connection = Connection(prefix=prefix)
    app_file_path = os.path.join(folder_path, collection_data.get("customer_apps"), f"{collection_data.get('customer_apps')}.json")
    app_data = CloneScriptHelper().read_json_file(app_file_path)
    category_file_path = os.path.join(folder_path, collection_data.get("category_apps"))
    new_category_data = push_app_category(category_file_path, collection_data.get("category_apps"), user_details, project_id, prefix)
    dashboard_file_path = os.path.join(folder_path, collection_data.get("dashboard"))
    push_dashboard(dashboard_file_path, collection_data.get("dashboard"), user_details, project_id, prefix)
    category_file_path = os.path.join(folder_path, collection_data.get("category"))
    push_category(category_file_path, collection_data.get("category"), project_id, prefix)
    unit_id_mapping, parameter_id_mapping = get_new_tag_and_unit_ids(folder_path, collection_data, project_id, prefix)
    widget_file_path = os.path.join(folder_path, collection_data.get("widget"))
    push_widget(widget_file_path, collection_data.get("widget"), project_id, unit_id_mapping, parameter_id_mapping, prefix)
    for data in app_data:
        data["access_list"]["users"].append(user_id)
        data["project_id"] = project_id
        data['meta']['created_by'] = user_id
        data["meta"]['last_updated_by'] = user_id
        manipulate_app_data(data, "dashboard_id", user_id, project_id)
        data['app_category_id'] = new_category_data.get(data.get('app_category_id'))
        query = {"app_id": data.get("app_id")}
        connection.customer_apps.update_one(query, {"$set": data}, upsert=True)
        print("apps got pushed")


def push_data(folder_path, collection_data, user_id, project_id, prefix=None):
    connection = Connection()
    user_details = connection.user_conn.find_one({"user_id": user_id}, {"_id": 0})
    push_app(folder_path, collection_data, user_id, user_details, project_id, prefix)


def extract_app_zip(app_name, user_id, project_id, prefix=None):
    zip_path = os.path.join(BASE_PATH, MOUNT_DIR, f"{app_name}.zip")
    folder_path = os.path.join(BASE_PATH, MOUNT_DIR, app_name)
    CloneScriptHelper().unzip_file(zip_path, folder_path)
    folders_list = CloneScriptHelper().list_folders(folder_path)
    folders_list.remove("app_image")
    folders_list.remove("app_logo")
    collection_names = {}
    for folder in folders_list:
        collection_names[folder.split(".")[1]] = folder
    push_data(folder_path, collection_names, user_id, project_id, prefix)


if __name__ == "__main__":
    extract_app_zip(APP_NAME, USER_ID, PROJECT_ID, PREFIX)
