"""
event_service.py

This script initializes and configures the FastAPI
 application for the Vision App: On Demand Service.

The application includes routes, middleware, and server config.

Usage:
    from scripts.core.services.event_service import router

"""
import json

from fastapi import APIRouter
from fastapi.responses import JSONResponse
from scripts.constants import Constants, APIConstants
from scripts.core.logging.application_logging import logger
from scripts.core.schemas.api import (
    trends_request, trends_response,
    trends_form_request, trends_form_response
)
from scripts.core.handler.trends_handler import TrendsDashboardHandler

downday_router = APIRouter(tags=[Constants.TRENDS_HANDLING_ENDPOINT])


@downday_router.on_event(APIConstants.SHUTDOWN)
async def shutdown_event():
    """
    This function prints the `Exiting!` string to the console.
    """
    print("Exiting!")


@downday_router.get(APIConstants.HEALTH_CHECK)
async def ping():
    """
    This function returns the `200 status message`.
    """
    return dict(status=200)


@downday_router.post(APIConstants.DASHBOARD_METADATA_ENDPOINT,
                     response_model=trends_response)
async def dashboard_metadata(
        request_data: trends_request
):
    """
    Initiate postgres db and create tables

    Args:
        request_data (DashboardFilterInput): Request body containing the
        necessary form parameters.

    Returns:
    """
    try:
        trends_obj = TrendsDashboardHandler()
        return trends_obj.get_trends_metadata(request_data=request_data)
    except Exception as err:
        logger.exception(
            Constants.EXCEPTION_RAISER.format(str(err)),
            exc_info=True,
        )
        return JSONResponse(status_code=500, content=str(err))


@downday_router.post(APIConstants.DASHBOARD_DATA_ENDPOINT,
                     response_model=trends_form_response)
async def dashboard_data(
        request_data: trends_form_request
):
    """
    Initiate postgres db and create tables

    Args:
        request_data (DashboardFormInput): Request body containing the
        necessary form parameters.

    Returns:
    """
    try:
        trends_obj = TrendsDashboardHandler()
        return trends_obj.get_trends_data(request_data=request_data)
    except Exception as err:
        logger.exception(
            Constants.EXCEPTION_RAISER.format(str(err)),
            exc_info=True,
        )
        return JSONResponse(status_code=500, content=str(err))
