"""
Project: Trends Dashboard Handler

Description:
    This module contains a class responsible for creating tables and performing
    calculations for PepsiCo metrics.
    It provides methods to retrieve metadata and data for the Trends dashboard.

Author: [H115-301: Suryakant Soni]

"""

from scripts.configurations import postgres_details
from scripts.constants import Constants
from scripts.core.db.postgres import database_init
from scripts.core.db.postgres.psql_query import (
    get_departments, get_parameters, get_filter_query, get_form_data
)
from scripts.core.schemas.postgres import TableObject
from scripts.core.logging.application_logging import logger
from scripts.core.schemas.postgres.postgres_tables import \
    TrendsMasterTable, TrendsFormTable


class TrendsDashboardHandler:
    """
    Class responsible for creating tables and performing calculations for
    PepsiCo metrics.
    """

    def get_trends_metadata(self, request_data):
        """
        Retrieve metadata for trends dashboard.

        Parameters:
            request_data : Object containing request parameters.

        Returns:
            dict: Metadata for trends dashboard.
        """
        logger.info("Getting Metadata")
        final_metadata_json = dict()
        clause = None

        logger.info("Database initialization")
        db_init = database_init()

        logger.info("Initiating table object")
        trends_master_tbl_obj = TableObject(
            db=db_init, table_name=TrendsMasterTable
        )

        logger.info(f"Current level: {request_data.current_level}")

        if request_data.current_level or request_data.current_level == 0:
            final_metadata_json["current_level"] = request_data.current_level + 1
            filter_query, clause = get_filter_query(
                current_level=request_data.current_level,
                department=request_data.department,
                function=request_data.function,
                form=request_data.form,
                filter_value=request_data.filter,
                table=postgres_details.master_tbl
            )
        else:
            final_metadata_json["current_level"] = 0
            filter_query = get_departments(
                table=TrendsMasterTable
            )

        logger.info("Getting response from the Trends Master Table")
        response_data = trends_master_tbl_obj.execute_query(
            query=filter_query
        )
        parameter_flag = False

        if response_data:
            final_metadata_json, parameter_flag = self.get_response_data(
                response_data=response_data,
                final_metadata_json=final_metadata_json,
                parameter_flag=parameter_flag
            )

        if parameter_flag:
            filter_query = get_parameters(table=postgres_details.master_tbl,
                                          where_clause=clause)

            # Getting response from the Trends Master Table
            response_data = trends_master_tbl_obj.execute_query(
                query=filter_query
            )
            if response_data:
                final_metadata_json, parameter_flag = self.get_response_data(
                    response_data=response_data,
                    final_metadata_json=final_metadata_json,
                    parameter_flag=parameter_flag
                )
        return final_metadata_json

    def get_trends_data(self, request_data):
        """
        Retrieve data for trends dashboard.

        Parameters:
            request_data : Object containing request parameters.

        Returns:
            dict: Data for trends dashboard.
        """
        logger.info("Getting data from form table")
        final_data_json = dict()

        logger.info("Database initialization")
        db_init = database_init()

        # Creating table object
        trends_data_tbl_obj = TableObject(
            db=db_init, table_name=TrendsFormTable
        )

        if request_data.start_date and request_data.end_date \
                and request_data.mapping_id:
            data_query = get_form_data(
                metadata_tbl=postgres_details.master_tbl,
                data_tbl=postgres_details.data_tbl,
                mapping_id=request_data.mapping_id,
                start_date=request_data.start_date,
                end_date=request_data.end_date
            )
            # Getting response from the Trends Master Table
            response_data = trends_data_tbl_obj.execute_query(
                query=data_query
            )
            if response_data:
                logger.info("Calculate mean value")
                total_value = sum(entry['value'] for entry in response_data)
                mean_value = round((total_value / len(response_data)), 2)

                logger.info("Creating the required JSON format")
                final_data_json = {
                    "header_content": Constants.HEADER_CONTENT,
                    "body_content": [
                        {
                            "date": entry['time_stamp'].strftime("%m-%d-%Y %H:%M:%S"),
                            "parameter": entry['parameter'],
                            "value": entry['value'],
                            "mean": mean_value,
                            "upper_limit": entry['upper_value'],
                            "lower_limit": entry['lower_limit']
                        }
                        for entry in response_data
                    ],
                    "upper_limit": response_data[0]['upper_value'],
                    "lower_limit": response_data[0]['lower_limit'],
                    "mean": mean_value
                }
            else:
                logger.info("No data in database")
                final_data_json = {
                    "header_content": Constants.HEADER_CONTENT,
                    "body_content": [],
                    "upper_limit": 0,
                    "lower_limit": 0,
                    "mean": 0
                }

        return final_data_json

    def get_response_data(self, response_data, final_metadata_json, parameter_flag):
        """
        Process response data.

        Parameters:
            response_data (list): List of response data.
            final_metadata_json (dict): Final metadata JSON.
            parameter_flag (bool): Flag indicating parameter presence.

        Returns:
            tuple: Tuple containing updated metadata JSON and parameter flag.
        """
        values = []

        if parameter_flag:
            logger.info("Getting parameter data from metadata table")
            filter_data = {}

            # Iterate through the response_data list
            for each_data in response_data:
                parameter_name = each_data['parameter']
                parameter_type = each_data['parameter_type']

                # If the parameter doesn't exist in filter_data, add it
                if parameter_name not in filter_data:
                    filter_data[parameter_name] = {
                        "key": parameter_name,
                        "label": parameter_name,
                        "mapping_id": each_data["mapping_id"],
                        "type": []
                    }

                # Add the parameter_type to the type list if it doesn't exist
                if parameter_type not in [t["key"] for t in
                                          filter_data[parameter_name]["type"]]:
                    filter_data[parameter_name]["type"].append({
                        "key": parameter_type,
                        "label": parameter_type
                    })
            final_metadata_json.update({
                "filter_name": "parameter",
                "values": list(filter_data.values())
            })
        else:
            logger.info("Getting filter values from metadata table")
            for each_data in response_data:
                if not parameter_flag and len(each_data) > 1:
                    filter_name = list(each_data.items())
                    if filter_name[1][1]:
                        values.append({
                            "key": filter_name[1][1],
                            "label": filter_name[1][1]
                        })
                    else:
                        parameter_flag = True
                        break
                    final_metadata_json["filter_name"] = filter_name[0][1]
                    final_metadata_json["values"] = values
                elif not parameter_flag and len(each_data) == 1:
                    filter_name = list(each_data.keys())[0]
                    values.append({
                        "key": each_data[filter_name],
                        "label": each_data[filter_name]
                    })
                    final_metadata_json["filter_name"] = filter_name
                    final_metadata_json["values"] = values
        return final_metadata_json, parameter_flag
