import json
import requests
from fastapi import Request, Response
from fastapi.openapi.models import APIKey, APIKeyIn
from fastapi.security import APIKeyCookie
from fastapi.security.api_key import APIKeyBase

from scripts.configurations import security_conf_details
from scripts.core.logging.application_logging import logger
from scripts.core.security.apply_encryption_util import create_token


class CommonUtils:
    @staticmethod
    def decode_params(params: str):
        # This will decode the parameters to dictionary format and return it back.
        return json.loads(params.strip("'\\"))


class AppName(APIKeyBase):
    """
    Authentication backend using a cookie.
    Internally, uses a JWT token to store the data.
    """

    scheme: APIKeyCookie
    cookie_name: str
    cookie_secure: bool

    def __init__(
        self,
        cookie_name: str = "app",
    ):
        super().__init__()
        self.model: APIKey = APIKey(**{"in": APIKeyIn.cookie}, name=cookie_name)
        self.scheme_name = self.__class__.__name__
        self.cookie_name = cookie_name
        self.scheme = APIKeyCookie(name=self.cookie_name, auto_error=False)

    async def __call__(self, request: Request, response: Response) -> str:
        cookies = request.cookies
        app = cookies.get("app")
        if not app:
            app = request.headers.get("app")
        if app:
            response.set_cookie(
                "app",
                app,
                samesite="strict",
                httponly=True,
                max_age=6000000000 * 60,
            )
            response.headers["app"] = app
        return app


def create_internal_token(
    host: str = "127.0.0.1", user_id=None, internal_token=security_conf_details.token
):
    """
    This method is to create a cookie
    Check this to see the implementation: https://gitlab-pm.knowledgelens.com/KnowledgeLens/Products/iLens-2.0/core
    /server/ilens-assistant/workflow-management/-/blob/develop/scripts/core/handlers/task_handler.py#L365
    """

    try:
        if user_id is None:
            user_id = "user_099"
        return create_token(
            user_id=user_id,
            ip=host,
            token=internal_token,
        )
    except Exception as e:
        logger.exception(str(e))
        raise


def hit_external_service(
    request_url: str, method: str, request_payload: dict, headers: dict, **files
):
    try:
        logger.info(f"Hitting external service {request_url}")
        return requests.request(
            method=method,
            url=request_url,
            data=request_payload,
            headers=headers,
            timeout=10,
            **files,
        )

    except Exception as e:
        logger.exception(f"Exception while hit_external_service {e}")
        raise
