import logging as logger

import psycopg2
from psycopg2.extras import RealDictCursor

from scripts.configurations import postgres_details
from scripts.core.exception.app_exceptions import ErrorMessages, PostgresDBError


class PostgresDBUtility:
    def __init__(self):
        try:
            logger.debug("Meta DB connection initialization")
            self.host = postgres_details.host
            self.port = postgres_details.port
            self.username = postgres_details.username
            self.password = postgres_details.password
            self.db_name = postgres_details.database
            self.cursor_type = RealDictCursor
        except Exception as e:
            logger.error(f"Exception in the Meta DB initialization : {str(e)}")

    def create_connection(self, db=None):
        _db = db or self.db_name
        conn = psycopg2.connect(
            database=_db, user=self.username, password=self.password, host=self.host,
            port=self.port
        )
        logger.debug("Connection established successfully")
        return conn

    def select_postgres_table(self, query, db):
        """
        This method is used for selecting records from tables.
        :param query: The select query to be executed
        :param db: Session
        :return: status: The status True on success and False on failure and
        the list of rows
        """
        logger.debug(f" SQL QUERY {query}")
        connection = None
        result = ""
        flag = False
        try:
            connection = self.create_connection(db)
            with connection.cursor(cursor_factory=self.cursor_type) as cursor:
                cursor.execute(query)
                result = cursor.fetchall()
            flag = True
        except Exception as e:
            logger.error(f"Exception while fetching the data from db: {str(e)}")
        finally:
            try:
                if connection is not None:
                    connection.close()
            except Exception as e:
                logger.error(ErrorMessages.CONNECTION_EXCEPTION + str(e))
        return flag, result

    def select_postgres_fetchone(self, query):
        """
        This method is used for selecting records from tables.
        :param query: The select query to be executed
        :return: status: The status True on success and False on failure and
        the list of rows
        """
        connection = None
        result = ""
        flag = False
        try:
            connection = self.create_connection()
            logger.debug("Connection established successfully")

            with connection.cursor(cursor_factory=self.cursor_type) as cursor:
                cursor.execute(query)
                result = cursor.fetchone()
            flag = True
        except Exception as e:
            logger.error(f"Exception in select_postgres_fetchone: {str(e)}")
        finally:
            try:
                if connection is not None:
                    connection.close()
            except Exception as e:
                logger.error(ErrorMessages.CONNECTION_EXCEPTION + str(e))
        return flag, result

    def update_postgres_table(self, query, data=None, row_count_flag=False):
        """
        This method is used for updating tables.
        :param data: the list of values which needs to be updated
        :param query: The update query to be executed
        :param row_count_flag: Flag to return the row count
        :return: status: The status True on success and False on failure
        """
        logger.debug(" INSIDE SQL UPDATE")
        logger.debug(f" SQL QUERY {query}")
        logger.debug(f" VALUES {data}")
        connection = None
        flag = False
        try:
            connection = self.create_connection()
            with connection.cursor(cursor_factory=self.cursor_type) as cursor:
                cursor.execute(query, data)
                row_count = cursor.rowcount
            connection.commit()
            flag = True
        except Exception as e:
            logger.error(ErrorMessages.COMMON_MESSAGE + str(e))
        finally:
            try:
                if connection is not None:
                    connection.close()
            except Exception as e:
                logger.error(ErrorMessages.CONNECTION_EXCEPTION + str(e))
        return (flag, row_count) if row_count_flag else flag

    def fetch_list_data(self, query):
        list_of_lists = []
        connection = None
        try:
            connection = self.create_connection()
            cursor = connection.cursor()
            try:
                cursor.execute(query)
                json = cursor.fetchall()
                list_of_lists = list(
                    map(lambda each: [each["plant"], each["line"], each["machine"]],
                        json))
            except Exception as e:
                logger.error(f"Exception while fetching {str(e)}")
        except Exception as e:
            logger.error(f"Exception while connecting {str(e)}")
        finally:
            try:
                if connection is not None:
                    connection.close()
            except Exception as e:
                logger.error(ErrorMessages.CONNECTION_EXCEPTION + str(e))
        return list_of_lists

    def insert_postgresql_table(self, query, data=None, row_count_flag=False):
        """
        This method is used for inserting new records in tables.
        :param data: list of values which needs to be inserted
        :param query: The insert query to be executed
        :param row_count_flag: Flag to return the row count
        :return: status: The status True on success and False on failure
        """
        logger.debug(" INSIDE SQL INSERT")
        logger.debug(f" SQL QUERY {query}")
        logger.debug(f" VALUES {data}")
        connection = None
        flag = False
        try:
            connection = self.create_connection()
            with connection.cursor(cursor_factory=self.cursor_type) as cursor:
                cursor.execute(query, data)
                row_count = cursor.rowcount
            connection.commit()
            flag = True
        except Exception as e:
            logger.error(f"Exception while inserting: {str(e)}")
        finally:
            try:
                if connection is not None:
                    connection.close()
            except Exception as e:
                logger.error(ErrorMessages.CONNECTION_EXCEPTION + str(e))
        return (flag, row_count) if row_count_flag else flag

    def insert_many_postgresql_table(self, query, data=None):
        """
        This method is used for inserting new records in tables.
        :param data: list of values which needs to be inserted
        :param query: The insert query to be executed
        :return: status: The status True on success and False on failure
        """
        connection = None
        flag = False
        try:
            connection = self.create_connection()
            with connection.cursor(cursor_factory=self.cursor_type) as cursor:
                cursor.executemany(query, data)
            connection.commit()
            flag = True
        except Exception as e:
            logger.error(f"Exception while inserting: {str(e)}")
        finally:
            try:
                if connection is not None:
                    connection.close()
            except Exception as e:
                logger.error(ErrorMessages.CONNECTION_EXCEPTION + str(e))
        return flag

    def delete_postgres_table(self, query):
        """
        This method is used for updating tables.
        :param query: The update query to be executed
        :return: status: The status True on success and False on failure
        """
        connection = None
        flag = False
        try:
            connection = self.create_connection()
            with connection.cursor(cursor_factory=self.cursor_type) as cursor:
                cursor.execute(query)
            connection.commit()
            flag = True
        except Exception as e:
            logger.error(f"Exception while deleting: {str(e)}")
        finally:
            try:
                if connection is not None:
                    connection.close()
            except Exception as e:
                logger.error(f"Exception while closing connection: {str(e)}")
        return flag

    def check_table_existence(self, table_name, parameter_tuple):
        connection = None
        try:
            connection = self.create_connection()
            cursor = connection.cursor(cursor_factory=self.cursor_type)
            try:
                query = f""" CREATE TABLE IF NOT EXISTS {str(table_name)} {str(parameter_tuple)};"""

                cursor.execute(query)
            except Exception as e:
                logger.exception(f"Exception in the query execution{str(e)}")

        except Exception as e:
            logger.exception(
                f"Exception in the check table existance definition{str(e)}")

        finally:
            try:
                if connection is not None:
                    connection.close()
            except Exception as e:
                logger.error(ErrorMessages.CONNECTION_EXCEPTION + str(e))

    def fetch_data(self, query):
        cursor = None
        result = []
        try:
            connection = self.create_connection()
            cursor = connection.cursor(cursor_factory=self.cursor_type)
            cursor.execute(query)
            result = cursor.fetchall()
        except Exception as e:
            logger.error(ErrorMessages.COMMON_MESSAGE + str(e))
        finally:
            try:
                if cursor is not None:
                    cursor.close()
            except Exception as e:
                logger.error(ErrorMessages.CONNECTION_EXCEPTION + str(e))
        return result

    def connect(self, db):
        try:
            logger.debug(f"Establishing connection to DB Name: {db}")
            conn = self.create_connection(db)
            logger.debug(f"connection response: {conn}")
            return conn
        except Exception as e:
            logger.error(ErrorMessages.CONNECTION_EXCEPTION + str(e))
            raise PostgresDBError

    def fetch_table_data(self, query, db):
        cursor = None
        result = []
        try:
            connection = self.connect(db)
            cursor = connection.cursor()
            cursor.execute(query)
            result = cursor.fetchall()
            cursor.close()
            connection.close()
        except Exception as e:
            logger.error(ErrorMessages.COMMON_MESSAGE + str(e))
        finally:
            try:
                if cursor is not None:
                    cursor.close()
            except Exception as e:
                logger.error(ErrorMessages.CONNECTION_EXCEPTION + str(e))
        return result

    def fetch_schema_data(self, query, db):
        cursor = None
        result = []
        try:
            connection = self.create_connection(db)
            cursor = connection.cursor()
            cursor.execute(query)
            result = cursor.fetchall()
            cursor.close()
            connection.close()
        except Exception as e:
            logger.error(ErrorMessages.COMMON_MESSAGE + str(e))
        finally:
            try:
                if cursor is not None:
                    cursor.close()
            except Exception as e:
                logger.error(ErrorMessages.CONNECTION_EXCEPTION + str(e))
        return result
