"""
    event_service.py

    This script initializes and configures the FastAPI
     application for the Vision App: On Demand Service.

    The application includes routes, middleware, and server config.

Usage:
    from scripts.core.services.event_service import router

"""
from fastapi import APIRouter
from scripts.configurations import service_details
from scripts.core.handler.event_handler import CustomReportHandler
from scripts.core.logging.application_logging import logger
from scripts.constants import APIConstants, CommonConstants, figlet
from scripts.core.schemas.api import (
    custom_report_input_model,
    custom_report_output_model
)

report_handler_obj = CustomReportHandler()
event_router = APIRouter(tags=[CommonConstants.CUSTOM_REPORT_TAG])


@event_router.on_event(APIConstants.STARTUP)
async def startup_event():
    """
    This function prints the `figlet` string to the console.
    """
    print(figlet)


@event_router.on_event(APIConstants.SHUTDOWN)
async def shutdown_event():
    """
        This function prints the `Exiting!` string to the console.
    """
    print("Exiting!")


@event_router.get(
    APIConstants.HEALTH_CHECK)
async def ping():
    """
        This function returns the `200 status message`.
    """
    return {"status": 200}


@event_router.post(
    APIConstants.CUSTOM_REPORT_ENDPOINT, response_model=custom_report_output_model)
async def custom_report_function(input_json: custom_report_input_model):
    """
    Initiate postgres db and create tables

    Args:
        content (ReportInput): Request body containing the necessary parameters.

    Returns:
        :param :input_json

    """
    try:
        status, message, data = report_handler_obj.custom_report_handler(
            input_json=input_json
        )
        return {"status": status, "message": message, "data": data}
    except Exception as err:
        logger.exception(
            CommonConstants.EXCEPTION_RAISER.format(str(err)),
            exc_info=service_details.exception_trace,
        )
