import jwt
from jwt.exceptions import (
    ExpiredSignatureError,
    InvalidSignatureError,
    MissingRequiredClaimError,
)

from scripts.configurations import security_conf_details
from scripts.core.exception.app_exceptions import AuthenticationError, ErrorMessages
from scripts.core.logging.application_logging import logger


class JWT:
    def __init__(self):
        self.max_login_age = security_conf_details.lock_out_time_mins
        self.issuer = security_conf_details.issuer
        self.alg = security_conf_details.alg
        self.public = security_conf_details.public_key_path
        self.private = security_conf_details.private_key_path

    def encode(self, payload):
        try:
            with open(self.private) as f:
                key = f.read()
            return jwt.encode(payload, key, algorithm=self.alg)
        except Exception as e:
            logger.exception(f"Exception while encoding JWT: {str(e)}")
            raise
        finally:
            f.close()

    def validate(self, token):
        try:
            with open(self.public) as f:
                key = f.read()
            payload = jwt.decode(
                token,
                key,
                algorithms=self.alg,
                leeway=security_conf_details.leeway_in_mins,
                options={"require": ["exp", "iss"]},
            )
            return payload
        except InvalidSignatureError:
            raise AuthenticationError(ErrorMessages.ERROR003)
        except ExpiredSignatureError:
            raise AuthenticationError(ErrorMessages.ERROR002)
        except MissingRequiredClaimError:
            raise AuthenticationError(ErrorMessages.ERROR002)
        except Exception as e:
            logger.exception(f"Exception while validating JWT: {str(e)}")
            raise
        finally:
            f.close()
