import copy

import pandas as pd
from datetime import datetime
from scripts.constants import ReportType, CommonConstants, PostgresConstant
from scripts.template.ccpc_report_template import CCPCReportTemplate
from scripts.template.ccpp_report_template import CCPPReportTemplate
from scripts.template.refinery_report_template import SterliteRefineryTemplate
from scripts.template.acp_report_template import ACPReportTemplate
from scripts.core.logging.application_logging import logger
from scripts.core.exception.app_exceptions import GeneralException
from scripts.core.utilities.postgresql_db_utils import PostgresDBUtility


class CustomReportHandler:
    def __init__(self):
        self.postgres_db_obj = PostgresDBUtility()

    def create_custom_date_filter(self, input_json):
        """
        This method convert start date and end date to a date range.
        :param input_json:
        :return:
        """
        flag = False
        date_range_list = []

        if CommonConstants.PROPERTY in input_json and CommonConstants.START_DATE \
                in input_json[CommonConstants.PROPERTY] and CommonConstants.END_DATE \
                in input_json[CommonConstants.PROPERTY]:

            # Fetching Start date
            start_date = datetime.strptime(
                input_json[CommonConstants.PROPERTY][CommonConstants.START_DATE],
                CommonConstants.DATE_TIME_FORMAT
            )
            # Fetching End date
            end_date = datetime.strptime(
                input_json[CommonConstants.PROPERTY][CommonConstants.END_DATE],
                CommonConstants.DATE_TIME_FORMAT
            )

            logger.info(f"Creating list of dates starting from {start_date} "
                        f"to {end_date}")
            # Generating a date range
            date_list = pd.date_range(start=start_date,
                                      end=end_date,
                                      freq=CommonConstants.FREQUENCY)

            # Iterating and creating where clause filters
            for each_dates in date_list.strftime(
                    CommonConstants.DATE_TIME_FORMAT).to_list():

                # To get the each_dates - financial year
                financial_year = None
                date_obj = datetime.strptime(each_dates,
                                             CommonConstants.DATE_TIME_FORMAT)
                if date_obj.month >= 4:
                    financial_year = str(date_obj.year)
                elif date_obj.month < 4:
                    financial_year = str(date_obj.year - 1)

                date_range_list.append(
                    dict(
                        day_start_date=each_dates,
                        day_end_date=each_dates,
                        month_start_date=each_dates[:-2] + "01",
                        month_end_date=each_dates,
                        year_start_date=financial_year + "-04-01",
                        year_end_date=each_dates,
                    )
                )
            flag = True

        return flag, date_range_list

    def get_queries_from_db(self, input_json, date_filter):
        """
        :param input_json:
        :param date_filter:
        :return:
        """
        import numpy as np
        null_value = np.nan
        for each_blocks in input_json:
            # Iterating each blocks for fetching query


            if input_json[each_blocks][CommonConstants.QUERY]:
                for each_kpi in input_json[each_blocks][CommonConstants.QUERY]:
                    logger.info(f"KPI: {each_kpi}")

                    temp_data_dict = dict()
                    append_flag = False
                    data_list = []

                    database = PostgresConstant.STRELITE_DATALAKE_DEV
                    if "database" in input_json[each_blocks]:
                        database = input_json[each_blocks]["database"]

                    # Iterating each query for each KPI
                    if input_json[each_blocks][CommonConstants.QUERY][each_kpi]:
                        for each_query in \
                                input_json[each_blocks][CommonConstants.QUERY][
                                    each_kpi]:
                            query = each_query. \
                                format(
                                day_start_date=date_filter[
                                    CommonConstants.DAY_START_DATE],
                                day_end_date=date_filter[CommonConstants.DAY_END_DATE],
                                month_start_date=date_filter[
                                    CommonConstants.MONTH_START_DATE],
                                month_end_date=date_filter[
                                    CommonConstants.MONTH_END_DATE],
                                year_start_date=date_filter[
                                    CommonConstants.YEAR_START_DATE],
                                year_end_date=date_filter[
                                    CommonConstants.YEAR_END_DATE])

                            response = self.postgres_db_obj. \
                                select_postgres_table(query=query, db=database
                                                      )[1]

                            if response and "addition" not in input_json[each_blocks]:
                                if len(response) <= 1:
                                    temp_data_dict.update(dict(response[0]))
                                    append_flag = True
                                else:
                                    for every_data in response:
                                        input_json[each_blocks][CommonConstants.DATA]. \
                                            append(dict(every_data))

                            elif response and "addition" in input_json[each_blocks] and \
                                    "primary" in input_json[each_blocks]["addition"]:
                                for each_data in response:
                                    data_list.append(dict(each_data))
                                    append_flag = True

                            if not temp_data_dict and not append_flag:
                                # Creating null values if no data
                                for each_columns in input_json[each_blocks][
                                    CommonConstants.DATA_COLUMN]:
                                    temp_data_dict.update({each_columns: null_value})
                                append_flag = True

                        if append_flag and temp_data_dict:
                            input_json[each_blocks][CommonConstants.DATA].append(
                                temp_data_dict)
                    if data_list:
                        temp_json = {}

                        for item in data_list:
                            category = item[
                                input_json[each_blocks]["addition"]["primary"]]
                            if category not in temp_json:
                                temp_json[category] = {
                                    input_json[each_blocks]["addition"][
                                        "primary"]: category}

                            for key in item:
                                if key != input_json[each_blocks]["addition"][
                                    "primary"]:
                                    temp_json[category][key] = item[key]
                        for category, values in temp_json.items():
                            if values not in input_json[each_blocks][
                                CommonConstants.DATA]:
                                input_json[each_blocks][CommonConstants.DATA].append(
                                    values)
            else:
                temp_data_dict = dict()
                for each_columns in input_json[each_blocks][
                    CommonConstants.DATA_COLUMN]:
                    if each_columns:
                        temp_data_dict.update(
                            {each_columns: null_value}
                        )
                    else:
                        temp_data_dict.update(
                            {each_columns: ""}
                        )
                input_json[each_blocks][CommonConstants.DATA].append(temp_data_dict)
        return input_json

    def write_dataframe_to_excel(
            self,
            input_json, writer, workbook, sheet_name, start_col,
            start_row, header_merge_format, column_merge_format,
            blank_merge_format
    ):
        """
        :param input_json:
        :param writer:
        :param workbook:
        :param sheet_name:
        :param start_col:
        :param start_row:
        :param header_merge_format:
        :param column_merge_format:
        :param blank_merge_format:
        :return:
        """
        dataframes_list = []

        header_row = start_row
        border_json = dict()
        column_counter = 0
        previous_category_cols = 0
        header_flag = False
        color_json = dict()

        logger.info("Iterating through each block to create a dataframe")
        # Iterate through the categories and concatenate their data

        for category, category_data in input_json.items():
            data = category_data.get('data', [])
            data_frame = pd.DataFrame(data)
            data_frame.columns = category_data.get('columns', [])
            data_frame.dropna(how='all', inplace=True)
            dataframes_list.append(data_frame)
            shape = data_frame.shape

            # Checking if merged border in the JSON
            if "addition" in category_data and "merge_header" in category_data[
                "addition"]:
                header_flag = True
                header_start_col = previous_category_cols
                header_end_col = header_start_col + shape[1] - 1
                border_json[
                    (header_row, header_start_col, header_row, header_end_col)
                ] = category_data["addition"]["merge_header"]

            column_counter += shape[1]
            previous_category_cols = column_counter

            # Checking if color in addition section
            if "addition" in category_data and "column_color" in category_data[
                "addition"]:
                color_json.update(
                    category_data["addition"]["column_color"]
                )

        if header_flag:
            start_row += 1

        # Concatenate all DataFrames vertically (along rows)
        logger.info("Concatenating all dataframes to one")
        result_df = pd.concat(dataframes_list, axis=1, ignore_index=False)
        result_df.to_excel(
            writer, sheet_name=sheet_name,
            startcol=start_col, startrow=start_row + 1,
            header=False, index=False
        )

        # Get the xlsxwriter workbook and worksheet objects.
        worksheet = writer.sheets[sheet_name]

        # Generating merged header for each dataframe
        for merge_index, border_value in border_json.items():
            if border_value and border_value in color_json:
                merge_style_format = copy.deepcopy(header_merge_format)
                merge_style_format["fg_color"] = color_json[border_value]
                worksheet.merge_range(*merge_index, border_value,
                                      workbook.add_format(merge_style_format))
            else:
                worksheet.merge_range(*merge_index, border_value,
                                      workbook.add_format(header_merge_format))

        # Write the column headers with the defined format.
        for col_index, value in enumerate(result_df.columns.values):
            if value:
                if color_json and value in color_json:
                    style_format = copy.deepcopy(column_merge_format)
                    style_format["fg_color"] = color_json[value]
                    worksheet.write(
                        start_row, col_index, value,
                        workbook.add_format(style_format))
                else:
                    worksheet.write(
                        start_row, col_index, value,
                        workbook.add_format(column_merge_format))
            else:
                worksheet.write(
                    start_row, col_index, value,
                    workbook.add_format(blank_merge_format)
                )
        logger.info(f"Shape of current data frame is {result_df.shape}")

        return result_df.shape, worksheet, header_flag

    def custom_report_handler(self, input_json):
        """
        :param input_json:
        :return:
        """
        response_json = dict(
            job_id=input_json['job_id'],
            file_path="",
            file_name=input_json["file_name"],
            is_schedule="",
            project_id=input_json["project_id"]
        )
        try:
            logger.info(f"Report: {str(input_json['job_type']).lower()}")

            if str(input_json["job_type"]).lower() == ReportType.REFINERY_REPORT:

                logger.info("Generating custom date filter with in the range")
                # Getting custom date range using start date and end date
                date_flag, date_filter = self.create_custom_date_filter(
                    input_json=input_json)

                if date_flag:
                    with pd.ExcelWriter(
                            input_json["file_name"],
                            engine="xlsxwriter") as writer:

                        for each_date_range in date_filter:
                            logger.info(f"date filter: {each_date_range}")

                            report_template = copy.deepcopy(
                                SterliteRefineryTemplate.REPORT_TEMPLATE
                            )
                            start_col = CommonConstants.START_COLUMN
                            start_row = CommonConstants.START_ROW
                            total_column = 0
                            workbook = writer.book
                            sheet_name = datetime.strptime(
                                each_date_range[CommonConstants.DAY_START_DATE],
                                "%Y-%m-%d").strftime("%d %b %Y")

                            worksheet = None

                            # Iterating over sterlite json file
                            for each_blocks in report_template:

                                logger.info("Fetching each KPI data from queries")
                                each_blocks = self.get_queries_from_db(
                                    input_json=each_blocks, date_filter=each_date_range
                                )

                                logger.info("Writing each block into excel sheet")

                                shape, worksheet, header_flag = \
                                    self.write_dataframe_to_excel(
                                        input_json=each_blocks,
                                        writer=writer,
                                        workbook=workbook,
                                        sheet_name=sheet_name,
                                        start_col=start_col,
                                        start_row=start_row,
                                        header_merge_format=SterliteRefineryTemplate.COLUMN_HEADER_FORMAT,
                                        column_merge_format=SterliteRefineryTemplate.COLUMN_HEADER_FORMAT,
                                        blank_merge_format=SterliteRefineryTemplate.BLANK_COLUMN_HEADER_FORMAT
                                    )

                                if total_column < shape[1]:
                                    total_column = shape[1]
                                start_row += shape[0] + 2

                                if header_flag:
                                    start_row += 1

                            logger.info("Creating Header for each sheet")
                            self.merge_worksheet(
                                start_row=1, start_col=0, end_row=0,
                                end_col=0, title=sheet_name,
                                style=SterliteRefineryTemplate.WORKSHEET_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )
                            self.merge_worksheet(
                                start_row=1, start_col=1, end_row=0,
                                end_col=total_column - 1,
                                title=ReportType.REFINERY_WORKSHEET_NAME,
                                style=SterliteRefineryTemplate.WORKSHEET_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )

                            # Setting width to the column
                            worksheet.set_column(0, total_column,
                                                 CommonConstants.REFINERY_COLUMN_WIDTH)

                            # Center alignment of Excel data
                            workbook.formats[0].set_align('center')
                            logger.info(f"Report completed for: {each_date_range}")
                else:
                    logger.error("Missing: Start Date or End Date")

            if str(input_json["job_type"]).lower() == ReportType.ACP_REPORT:

                logger.info("Generating custom date filter with in the range")

                # Getting custom date range using start date and end date
                date_flag, date_filter = self.create_custom_date_filter(
                    input_json=input_json
                )

                if date_flag:
                    with pd.ExcelWriter(
                            input_json["file_name"],
                            engine="xlsxwriter") as writer:

                        for each_date_range in date_filter:
                            logger.info(f"date filter: {each_date_range}")

                            report_template = copy.deepcopy(
                                ACPReportTemplate.REPORT_TEMPLATE
                            )
                            start_col = CommonConstants.START_COLUMN
                            start_row = CommonConstants.START_ROW
                            total_column = 0
                            workbook = writer.book
                            sheet_name = datetime.strptime(
                                each_date_range[CommonConstants.DAY_START_DATE],
                                "%Y-%m-%d").strftime("%d %b %Y")

                            worksheet = None

                            # Iterating over sterlite json file
                            for each_blocks in report_template:

                                logger.info("Fetching each KPI data from queries")
                                each_blocks = self.get_queries_from_db(
                                    input_json=each_blocks, date_filter=each_date_range
                                )

                                logger.info("Writing each block into excel sheet")

                                shape, worksheet, header_flag = \
                                    self.write_dataframe_to_excel(
                                        input_json=each_blocks, writer=writer,
                                        workbook=workbook, sheet_name=sheet_name,
                                        start_col=start_col, start_row=start_row,
                                        header_merge_format=ACPReportTemplate.COLUMN_HEADER_FORMAT,
                                        column_merge_format=ACPReportTemplate.COLUMN_HEADER_FORMAT,
                                        blank_merge_format=ACPReportTemplate.BLANK_COLUMN_HEADER_FORMAT
                                    )

                                if total_column < shape[1]:
                                    total_column = shape[1]
                                start_row += shape[0] + 3

                                if header_flag:
                                    start_row += 1
                            logger.info("Creating Header for each sheet")

                            # -------- Image Section --------
                            logger.info("Adding Image in header")
                            self.merge_worksheet(
                                start_row=0, start_col=0, end_row=2, end_col=2,
                                title="",
                                style=ACPReportTemplate.WORKSHEET_IMAGE_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )
                            worksheet.insert_image('A1', ReportType.LOGO_IMAGE_PATH)

                            # -------- Heading Section --------
                            logger.info("Adding Heading in Header")
                            self.merge_worksheet(
                                start_row=0, start_col=3, end_row=1,
                                end_col=total_column - 1,
                                title=ReportType.ACP_WORKSHEET_NAME_1,
                                style=ACPReportTemplate.WORKSHEET_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )
                            self.merge_worksheet(
                                start_row=2, start_col=3,
                                end_row=2, end_col=total_column - 1,
                                title=ReportType.ACP_WORKSHEET_NAME_2,
                                style=ACPReportTemplate.WORKSHEET_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )

                            # -------- Date HEADER Section --------
                            logger.info("Adding Date time in header")
                            self.merge_worksheet(
                                start_row=3, start_col=0, end_row=3, end_col=1,
                                title=ReportType.DATE,
                                style=ACPReportTemplate.WORKSHEET_DATE_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )

                            # -------- Date Section --------
                            logger.info("Adding Date time in header")
                            self.merge_worksheet(
                                start_row=3, start_col=2,
                                end_row=3, end_col=3, title=sheet_name,
                                style=ACPReportTemplate.WORKSHEET_DATE_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )

                            # Setting width to the column
                            worksheet.set_column(0, total_column,
                                                 CommonConstants.ACP_COLUMN_WIDTH)

                            # Center alignment of Excel data
                            workbook.formats[0].set_align('center')
                            logger.info(f"Report completed for: {each_date_range}")
                else:
                    logger.error("Missing: Start Date or End Date")

            if str(input_json["job_type"]).lower() == ReportType.CCPC_REPORT:

                logger.info("Generating custom date filter with in the range")

                # Getting custom date range using start date and end date
                date_flag, date_filter = self.create_custom_date_filter(
                    input_json=input_json
                )

                if date_flag:
                    with pd.ExcelWriter(
                            input_json["file_name"],
                            engine="xlsxwriter") as writer:

                        for each_date_range in date_filter:
                            logger.info(f"date filter: {each_date_range}")

                            report_template = copy.deepcopy(
                                CCPCReportTemplate.CCPC_REPORT_TEMPLATE
                            )
                            start_col = CommonConstants.START_COLUMN
                            start_row = CommonConstants.START_ROW
                            total_column = 0
                            workbook = writer.book
                            sheet_name = datetime.strptime(
                                each_date_range[CommonConstants.DAY_START_DATE],
                                "%Y-%m-%d").strftime("%d %b %Y")

                            worksheet = None

                            # Iterating over sterlite json file
                            for each_blocks in report_template:

                                logger.info("Fetching each KPI data from queries")
                                each_blocks = self.get_queries_from_db(
                                    input_json=each_blocks, date_filter=each_date_range
                                )

                                logger.info("Writing each block into excel sheet")

                                shape, worksheet, header_flag = \
                                    self.write_dataframe_to_excel(
                                        input_json=each_blocks, writer=writer,
                                        workbook=workbook, sheet_name=sheet_name,
                                        start_col=start_col, start_row=start_row,
                                        header_merge_format=CCPCReportTemplate.COLUMN_HEADER_FORMAT,
                                        column_merge_format=CCPCReportTemplate.COLUMN_HEADER_FORMAT,
                                        blank_merge_format=CCPCReportTemplate.BLANK_COLUMN_HEADER_FORMAT
                                    )

                                if total_column < shape[1]:
                                    total_column = shape[1]
                                start_row += shape[0] + 2

                                if header_flag:
                                    start_row += 1
                            logger.info("Creating Header for each sheet")

                            # -------- Image Section --------
                            logger.info("Adding Image in header")
                            self.merge_worksheet(
                                start_row=0, start_col=0, end_row=2, end_col=2,
                                title="",
                                style=ACPReportTemplate.WORKSHEET_IMAGE_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )
                            worksheet.insert_image('A1', ReportType.LOGO_IMAGE_PATH)

                            # -------- Heading Section --------
                            logger.info("Adding Heading in Header")
                            self.merge_worksheet(
                                start_row=0, start_col=3, end_row=1,
                                end_col=total_column - 1,
                                title=ReportType.ACP_WORKSHEET_NAME_1,
                                style=CCPCReportTemplate.WORKSHEET_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )
                            self.merge_worksheet(
                                start_row=2, start_col=3,
                                end_row=2, end_col=total_column - 1,
                                title=ReportType.ACP_WORKSHEET_NAME_2,
                                style=CCPCReportTemplate.WORKSHEET_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )

                            # -------- Date HEADER Section --------
                            logger.info("Adding Date time in header")
                            self.merge_worksheet(
                                start_row=3, start_col=0, end_row=3, end_col=1,
                                title=ReportType.DATE,
                                style=ACPReportTemplate.WORKSHEET_DATE_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )

                            # -------- Date Section --------
                            logger.info("Adding Date time in header")
                            self.merge_worksheet(
                                start_row=3, start_col=2,
                                end_row=3, end_col=3, title=sheet_name,
                                style=ACPReportTemplate.WORKSHEET_DATE_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )

                            # Setting width to the column
                            worksheet.set_column(0, total_column,
                                                 CommonConstants.ACP_COLUMN_WIDTH)

                            # Hide gridlines from the excel sheet
                            # worksheet.hide_gridlines(2)

                            # Center alignment of Excel data
                            workbook.formats[0].set_align('center')
                            logger.info(f"Report completed for: {each_date_range}")
                else:
                    logger.error("Missing: Start Date or End Date")

            if str(input_json["job_type"]).lower() == ReportType.CCPP_REPORT:

                logger.info("Generating custom date filter with in the range")

                # Getting custom date range using start date and end date
                date_flag, date_filter = self.create_custom_date_filter(
                    input_json=input_json
                )

                if date_flag:
                    with pd.ExcelWriter(
                            input_json["file_name"],
                            engine="xlsxwriter") as writer:

                        for each_date_range in date_filter:
                            logger.info(f"date filter: {each_date_range}")

                            report_template = copy.deepcopy(
                                CCPPReportTemplate.CCPP_REPORT_TEMPLATE
                            )
                            start_col = CommonConstants.START_COLUMN
                            start_row = CommonConstants.START_ROW
                            total_column = 0
                            workbook = writer.book
                            sheet_name = datetime.strptime(
                                each_date_range[CommonConstants.DAY_START_DATE],
                                "%Y-%m-%d").strftime("%d %b %Y")

                            worksheet = None

                            # Iterating over sterlite json file
                            for each_blocks in report_template:

                                logger.info("Fetching each KPI data from queries")
                                each_blocks = self.get_queries_from_db(
                                    input_json=each_blocks, date_filter=each_date_range
                                )

                                logger.info("Writing each block into excel sheet")

                                shape, worksheet, header_flag = \
                                    self.write_dataframe_to_excel(
                                        input_json=each_blocks, writer=writer,
                                        workbook=workbook, sheet_name=sheet_name,
                                        start_col=start_col, start_row=start_row,
                                        header_merge_format=CCPPReportTemplate.COLUMN_HEADER_FORMAT,
                                        column_merge_format=CCPPReportTemplate.COLUMN_HEADER_FORMAT,
                                        blank_merge_format=CCPPReportTemplate.BLANK_COLUMN_HEADER_FORMAT
                                    )

                                if total_column < shape[1]:
                                    total_column = shape[1]
                                start_row += shape[0] + 3

                                if header_flag:
                                    start_row += 1
                            logger.info("Creating Header for each sheet")

                            # -------- Image Section --------
                            logger.info("Adding Image in header")
                            self.merge_worksheet(
                                start_row=0, start_col=0, end_row=2, end_col=2,
                                title="",
                                style=ACPReportTemplate.WORKSHEET_IMAGE_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )
                            worksheet.insert_image('A1', 'vedanta_sterlite_copper.png')

                            # -------- Heading Section --------
                            logger.info("Adding Heading in Header")
                            self.merge_worksheet(
                                start_row=0, start_col=3, end_row=1,
                                end_col=total_column - 1,
                                title=ReportType.ACP_WORKSHEET_NAME_1,
                                style=ACPReportTemplate.WORKSHEET_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )
                            self.merge_worksheet(
                                start_row=2, start_col=3,
                                end_row=2, end_col=total_column - 1,
                                title=ReportType.ACP_WORKSHEET_NAME_2,
                                style=ACPReportTemplate.WORKSHEET_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )

                            # -------- Date HEADER Section --------
                            logger.info("Adding Date time in header")
                            self.merge_worksheet(
                                start_row=3, start_col=0, end_row=3, end_col=1,
                                title=ReportType.DATE,
                                style=ACPReportTemplate.WORKSHEET_DATE_HEADER_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )

                            # -------- Date Section --------
                            logger.info("Adding Date time in header")
                            self.merge_worksheet(
                                start_row=3, start_col=2,
                                end_row=3, end_col=3, title=sheet_name,
                                style=ACPReportTemplate.WORKSHEET_DATE_FORMAT,
                                worksheet=worksheet, workbook=workbook,
                            )

                            # Setting width to the column
                            worksheet.set_column(0, total_column,
                                                 CommonConstants.ACP_COLUMN_WIDTH)

                            # Center alignment of Excel data
                            workbook.formats[0].set_align('center')
                            logger.info(f"Report completed for: {each_date_range}")
                else:
                    logger.error("Missing: Start Date or End Date")

        except GeneralException as err:
            logger.error(f"Exception in custom_report_handler: {err}")
        return response_json

    def merge_worksheet(self, start_row, start_col,
                        end_row, end_col, title, style, worksheet, workbook):
        """
        This method is to merge the worksheet
        :param start_row:
        :param start_col:
        :param end_row:
        :param end_col:
        :param title:
        :param style:
        :param worksheet:
        :param workbook:
        :return:
        """
        style_format = workbook.add_format(style)
        worksheet.merge_range(
            start_row, start_col, end_row, end_col, title, style_format
        )
