import copy

import pandas as pd
from datetime import datetime
from scripts.constants import ReportType, CommonConstants
from scripts.template.sterlite_report_template import SterliteRefineryTemplate
from scripts.core.logging.application_logging import logger
from scripts.core.exception.app_exceptions import GeneralException
from scripts.core.utilities.postgresql_db_utils import PostgresDBUtility


class CustomReportHandler:
    def __init__(self):
        self.postgres_db_obj = PostgresDBUtility()

    def create_custom_date_filter(self, input_json):
        """
        This method convert start date and end date to a date range.
        :param input_json:
        :return:
        """
        date_range_list = []
        # Fetching Start date
        start_date = datetime.strptime(
            input_json[CommonConstants.PROPERTY][CommonConstants.START_DATE],
            CommonConstants.DATE_TIME_FORMAT
        )
        # Fetching End date
        end_date = datetime.strptime(
            input_json[CommonConstants.PROPERTY][CommonConstants.END_DATE],
            CommonConstants.DATE_TIME_FORMAT
        )

        logger.info(f"Creating list of dates starting from {start_date} to {end_date}")
        # Generating a date range
        date_list = pd.date_range(start_date, end_date, freq=CommonConstants.FREQUENCY)

        # Iterating and creating where clause filters
        for each_dates in date_list.strftime(
                CommonConstants.DATE_TIME_FORMAT).to_list():

            # To get the each_dates - financial year
            financial_year = None
            date_obj = datetime.strptime(each_dates, CommonConstants.DATE_TIME_FORMAT)
            if date_obj.month >= 4:
                financial_year = str(date_obj.year)
            elif date_obj.month < 4:
                financial_year = str(date_obj.year - 1)

            date_range_list.append(
                dict(
                    day_start_date=each_dates,
                    day_end_date=each_dates,
                    month_start_date=each_dates[:-2] + "01",
                    month_end_date=each_dates,
                    year_start_date=financial_year + "-04-01",
                    year_end_date=each_dates,
                )
            )
        return date_range_list

    def get_queries_from_db(self, input_json, date_filter):
        """
        :param input_json:
        :param date_filter:
        :return:
        """
        for each_blocks in input_json:
            # Iterating each blocks for fetching query

            if input_json[each_blocks][CommonConstants.QUERY]:
                for each_kpi in input_json[each_blocks][CommonConstants.QUERY]:
                    logger.info(f"KPI: {each_kpi}")

                    temp_data_dict = dict()
                    append_flag = False

                    # Iterating each query for each KPI
                    if input_json[each_blocks][CommonConstants.QUERY][each_kpi]:
                        for each_query in \
                                input_json[each_blocks][CommonConstants.QUERY][
                                    each_kpi]:
                            query = each_query. \
                                format(
                                day_start_date=date_filter[
                                    CommonConstants.DAY_START_DATE],
                                day_end_date=date_filter[CommonConstants.DAY_END_DATE],
                                month_start_date=date_filter[
                                    CommonConstants.MONTH_START_DATE],
                                month_end_date=date_filter[
                                    CommonConstants.MONTH_END_DATE],
                                year_start_date=date_filter[
                                    CommonConstants.YEAR_START_DATE],
                                year_end_date=date_filter[
                                    CommonConstants.YEAR_END_DATE])

                            response = self.postgres_db_obj.fetch_data(query=query)
                            logger.info(f"Response length: {len(response)}")
                            if response:
                                if len(response) <= 1:
                                    temp_data_dict.update(dict(response[0]))
                                    append_flag = True
                                else:
                                    for every_data in response:
                                        input_json[each_blocks][CommonConstants.DATA]. \
                                            append(dict(every_data))

                            if not temp_data_dict and not append_flag:
                                # Creating null values if no data
                                for each_columns in input_json[each_blocks][
                                    CommonConstants.DATA_COLUMN]:
                                    temp_data_dict.update({each_columns: None})
                                append_flag = True

                        if append_flag:
                            input_json[each_blocks][CommonConstants.DATA].append(
                                temp_data_dict)
            else:
                temp_data_dict = dict()
                for each_columns in input_json[each_blocks][
                    CommonConstants.DATA_COLUMN]:
                    temp_data_dict.update(
                        {each_columns: None}
                    )
                input_json[each_blocks][CommonConstants.DATA].append(temp_data_dict)
        return input_json

    def write_dataframe_to_excel(
            self,
            input_json,
            writer,
            workbook,
            sheet_name,
            start_col,
            start_row
    ):
        """
        :param input_json:
        :param writer:
        :param workbook:
        :param sheet_name:
        :param start_col:
        :param start_row:
        :return:
        """
        dataframes_list = []

        logger.info("Iterating through each block to create a dataframe")
        # Iterate through the categories and concatenate their data
        for category, category_data in input_json.items():
            data = category_data.get('data', [])
            data_frame = pd.DataFrame(data)
            data_frame.columns = category_data.get('columns', [])
            dataframes_list.append(data_frame)

        # Concatenate all DataFrames vertically (along rows)
        logger.info("Concatenating all dataframes to one")
        result_df = pd.concat(dataframes_list, axis=1, ignore_index=False)
        result_df.to_excel(
            writer, sheet_name=sheet_name,
            startcol=start_col, startrow=start_row + 1,
            header=False, index=False
        )

        # Get the xlsxwriter workbook and worksheet objects.
        worksheet = writer.sheets[sheet_name]

        # Write the column headers with the defined format.
        for col_index, value in enumerate(result_df.columns.values):
            worksheet.write(
                start_row, col_index, value,
                workbook.add_format(
                    SterliteRefineryTemplate.COLUMN_HEADER_FORMAT)
                )
        logger.info(f"Shape of current data frame is {result_df.shape}")

        return result_df.shape, worksheet

    def custom_report_handler(self, input_json):
        """
        :param input_json:
        :return:
        """
        status = False
        message = "Error generating a message"
        data = input_json["file_name"]
        try:
            # if str(input_json.job_type).lower() == ReportType.REFINERY_REPORT:
            if str(input_json["job_type"]).lower() == ReportType.REFINERY_REPORT:

                logger.info("Generating custom date filter with in the range")
                # Getting custom date range using start date and end date
                date_filter = self.create_custom_date_filter(input_json=input_json)

                with pd.ExcelWriter(
                        input_json["file_name"],
                        engine="xlsxwriter") as writer:

                    for each_date_range in date_filter:
                        logger.info(f"date filter: {each_date_range}")

                        report_template = copy.deepcopy(
                            SterliteRefineryTemplate.REPORT_TEMPLATE
                        )
                        start_col = CommonConstants.START_COLUMN
                        start_row = CommonConstants.START_ROW
                        total_column = 0
                        workbook = writer.book
                        sheet_name = each_date_range[
                            CommonConstants.DAY_START_DATE]
                        worksheet = None

                        # Iterating over sterlite json file
                        for each_blocks in report_template:

                            logger.info("Fetching each KPI data from queries")
                            each_blocks = self.get_queries_from_db(
                                input_json=each_blocks, date_filter=each_date_range
                            )

                            logger.info(f"start_row: {start_row}")
                            logger.info("Writing each block into excel sheet")

                            shape, worksheet = self.write_dataframe_to_excel(
                                input_json=each_blocks,
                                writer=writer,
                                workbook=workbook,
                                sheet_name=sheet_name,
                                start_col=start_col,
                                start_row=start_row,

                            )
                            logger.info(f"Last row: {shape[0]}")
                            logger.info(f"Last column: {shape[1]}")

                            if total_column < shape[1]:
                                total_column = shape[1]
                            start_row += shape[0] + 2
                            logger.info(f"end_row: {start_row}")

                        # Add a header format.
                        main_header_format = workbook.add_format(
                            SterliteRefineryTemplate.WORKSHEET_HEADER_FORMAT)

                        logger.info("Creating Header for each sheet")

                        # (Merge Rows, Start Column, '', Total Column, Title, Format)
                        worksheet.merge_range(1, 0, 0, 0, sheet_name,
                                              main_header_format)
                        worksheet.merge_range(1, 1, 0, total_column - 1,
                                              ReportType.REFINERY_WORKSHEET_NAME,
                                              main_header_format)
                        logger.info(f"Report completed for: {each_date_range}")

        except GeneralException as err:
            logger.error(f"Exception in custom_report_handler: {err}")
        return status, message, data
