import datetime
import os
import traceback
from pathlib import Path

from azure.storage.blob import BlobServiceClient
from loguru import logger

from scripts.constants.app_configuration import BLOB_CONTAINER_NAME, BLOB_ACCOUNT_NAME, BLOB_ACCOUNT_KEY


class BlobUtil:
    """
    For downloading and uploading datas via blob storage
    """
    def __init__(self, project, site, line, camera):
        self.project = project
        self.site = site
        self.line = line
        self.camera = camera
        self.blob_service_client = BlobServiceClient(f"https://{BLOB_ACCOUNT_NAME}.blob.core.windows.net",
                                                     credential=BLOB_ACCOUNT_KEY)
        self.container = BLOB_CONTAINER_NAME

    def download(self, download_path, dest_path):
        try:
            download_file_path = os.path.join(dest_path, "dataset.zip")
            blob_client = self.blob_service_client.get_blob_client(container=self.container,
                                                                   blob=download_path)
            with open(download_file_path, "wb") as download_file:
                blob_client.download_blob().readinto(download_file)
            logger.info(f'download from Azure Storage with blob path:{download_path}')

            return True
        except Exception as e:
            traceback.print_exc()
            logger.error(f"Failed to Pull Files from blob Storage! : {str(e)}")
            return False

    def upload(self, local_path, file_name):
        try:
            upload_file_path = os.path.join(local_path, file_name)
            blob_client = self.blob_service_client.get_blob_client(container=self.container,
                                                                   blob=f'{self.project}/{self.site}/{self.line}/'
                                                                        f'{self.camera}/'
                                                                        f'{datetime.datetime.now().hour}_'
                                                                        f'{datetime.datetime.now().day}_'
                                                                        f'{datetime.datetime.now().month}_'
                                                                        f'{datetime.datetime.now().year}'
                                                                        f'/{file_name}')
            with open(file=upload_file_path, mode="rb") as data:
                blob_client.upload_blob(data)
            logger.info(
                f'Uploaded to Azure Storage with blob path:{self.project}/{self.site}/{self.line}/{self.camera}/'
                f'{datetime.datetime.now().hour}_{datetime.datetime.now().day}_{datetime.datetime.now().month}_{datetime.datetime.now().year}'
                f'/{file_name}')
            return True
        except Exception as e:
            traceback.print_exc()
            logger.error(f"Failed to Push Files to blob Storage! : {str(e)}")
            return False
