import datetime
import json

from scripts.db.mongo import mongo_client
from scripts.db.mongo.ilens_configuration.aggregates.tags import TagsAggregate
from scripts.db.redis_connection import rules_redis
from scripts.constants import RuleCommonKeys
from scripts.logging.logger import logger
from scripts.utils.common_utils import CommonUtils
from scripts.db.mongo.ilens_configuration.collections.rule_engine import RuleEngine
from scripts.db.mongo.ilens_configuration.collections.tags import Tags


class RuleConfigurationHandler(object):
    def __init__(self):
        self.rules_redis = rules_redis
        self.rules_mongo = RuleEngine(mongo_client=mongo_client)
        self.tags_mongo = Tags(mongo_client=mongo_client)
        self.tags_aggregate = TagsAggregate
        self.common_utils = CommonUtils()

    def create_rule_engine(self, input_data):
        """
        this will function will save rule meta data
        """
        try:
            user_id = self.common_utils.get_usr_id(input_data=input_data)
            if not user_id:
                return {"status": "failed", "message": "Unauthorized user"}
            new_rule = False
            if "rule_engine_id" in input_data and input_data["rule_engine_id"] == "":
                rule_name_exists = self.duplicate_rule_name_check(input_data=input_data)
                if rule_name_exists:
                    return {"status": "failed", "message": "Rule name already exists!!"}
                input_data["created_on"] = (datetime.datetime.now()).strftime("%d/%m/%Y")
                input_data["rule_engine_id"] = "rule_engine_" + self.common_utils.get_next_id("rule_engine")
                input_data["created_by"] = user_id
                new_rule = True
            input_data['disable_all'] = input_data.get('disable_all', False)
            insert_block_json_list = []
            schedule_start = None
            schedule_end = None
            schedule_type = 'realtime'
            execute_on = dict()
            rule_type = input_data['Selected_ruleType'].lower().replace(" ", "")
            if rule_type == RuleCommonKeys.SCHEDULE:
                schedule_data = input_data.get('schedule', dict)
                if schedule_data.get('schedule_Type') == 'recurring':
                    schedule_start = int(schedule_data['duration_startDate'])
                    schedule_end = int(schedule_data['duration_endDate'])
                schedule_type = schedule_data.get('schedule_Type', '')
                execute_on = self.get_execute_on(input_data.get("schedule", dict()))
            derived_tag_list = self.get_derived_tags()
            output_rule_block_map = self.get_rule_block_mapping(input_data['project_id'])
            if "calcFormulaList" in input_data:  # check if blocks exists or rule is empty
                counter = 1
                for each_block in input_data["calcFormulaList"]:  # Assign block id
                    each_block.update({"block_id": "block_" + str(counter)})
                    counter += 1
                for each_block_data in input_data["calcFormulaList"]:  # each block data of rule
                    rule_block_parent = []
                    json_to_redis = self.make_json_to_push_into_redis(rule_block_data=each_block_data,
                                                                      input_data=input_data)  # prepare redis store json
                    json_to_redis['schedule_start'] = schedule_start
                    json_to_redis['schedule_end'] = schedule_end
                    if json_to_redis["transformation_type"] == "route":
                        if len(json_to_redis["output_tags"]) > 1:
                            json_to_redis["multi_tag"] = True
                        else:
                            json_to_redis["multi_tag"] = False
                    else:
                        cur_block_output_tag = ""
                        if each_block_data['completeTagId'] != "":
                            if each_block_data['completeTagId'] not in derived_tag_list:
                                # output tag can only be derived tag
                                return {"status": "failed", "message": "Rule output should be derived tag"}
                            # check if current block output is output of any other block outside rule
                            cur_block_output_tag = f"{each_block_data['output_devices'][0]}$" \
                                                   f"{each_block_data['completeTagId']}"
                            if cur_block_output_tag in output_rule_block_map:
                                if output_rule_block_map[cur_block_output_tag]['id'].split(".")[0] != input_data[
                                        'rule_engine_id']:
                                    raise Exception(
                                        f"Rule output of {each_block_data['block_id']} is output of a block of"
                                        f" rule {output_rule_block_map[cur_block_output_tag]['name']}")
                        tags_used_in_rule_block = json_to_redis["selectedTags"]
                        for each_used_tag in tags_used_in_rule_block:
                            # if a derived tag is input tag, it should be output of some other rule (cyclic restriction)
                            tag_value = each_used_tag.split("$")[-1]
                            if tag_value in derived_tag_list:
                                if each_used_tag not in output_rule_block_map:
                                    raise Exception(f"Derived tag used in {each_block_data['block_id']}"
                                                    " is not output of any other rule")
                            # check if this tag is used as output tag of another block within same rule
                            for each_block in input_data["calcFormulaList"]:
                                if each_block['completeTagId'] == '':
                                    continue  # lookup skip
                                if each_block['block_id'] == each_block_data['block_id']:  # skip if it is current block
                                    continue
                                block_result_tag = f"{each_block['output_devices'][0]}$" \
                                                   f"{each_block['completeTagId']}"
                                # check if block output is output of other block in the current rule
                                if cur_block_output_tag == block_result_tag:
                                    raise Exception("More than one block has same output tag")
                                if each_used_tag == block_result_tag:
                                    rule_block_parent.append(f"{input_data['rule_engine_id']}."
                                                             f"{each_block['block_id']}")
                            # check if this tag is used as output tag of any block of another rule
                            if each_used_tag in output_rule_block_map:
                                rule_block_parent.append(output_rule_block_map[each_used_tag]['id'])
                        each_block_data.update({"rule_block_parent": list(set(rule_block_parent))})
                        json_to_redis.update({"rule_block_parent": list(set(rule_block_parent))})
                        if len(json_to_redis["selectedTags"]) > 1:
                            json_to_redis["multi_tag"] = True
                        else:
                            json_to_redis["multi_tag"] = False
                    json_to_redis['rule_type'] = rule_type
                    json_to_redis['execute_on'] = execute_on
                    each_block_data.update({"execute_on": execute_on})
                    each_block_data.update({"execute_on_tag": json_to_redis['execute_on_tag']})
                    json_to_redis['previousTags'] = each_block_data.get('previousTags', [])
                    json_to_redis['schedule_type'] = schedule_type
                    counter += 1
                    insert_block_json_list.append(json_to_redis)
            if not new_rule:  # for edit delete the existing rule and  create new
                delete_response = self.delete_rule_engine(input_data, delete_on_create=True)
                if delete_response['status'] != 'success':
                    return {"status": "failed", "message": "Failed to update rule!!"}
                input_data['created_on'] = delete_response.get('created_on')
            if rule_type.lower() == RuleCommonKeys.SCHEDULE:
                self.update_schedule_rule_to_redis(insert_block_json_list, execute_on, input_data['rule_engine_id'],
                                                   schedule_type)
            elif rule_type.lower().replace(" ", "") == RuleCommonKeys.REALTIME:
                self.update_realtime_rule_to_redis(insert_block_json_list)
            input_data["last_updated"] = (datetime.datetime.now()).strftime("%d/%m/%Y")
            input_data['execute_on'] = execute_on
            new_values = input_data
            self.rules_mongo.delete_one_rule({"rule_engine_id": input_data["rule_engine_id"]})
            self.rules_mongo.update_one_rule(query_dict={"rule_engine_id": input_data["rule_engine_id"]},
                                             data=new_values, upsert=True)
            logger.debug("Rule saved successfully")
            return {"status": "success", "message": "rule  saved successfully", "rule_id": input_data["rule_engine_id"]}
        except Exception as e:
            logger.exception("Exception in rule creation:" + str(e))
            return {"status": "failed", "message": str(e)}

    def get_rule_block_mapping(self, project_id):
        output_rule_block_map = {}
        try:
            query = {"$or": [{"transformation_type": "validation_and_transformation"},
                             {"Selected_ruleType": "Schedule"}]}
            available_rules = self.rules_mongo.find_all_rules(query)
            for each_rule in available_rules:
                if each_rule.get('project_id', '') != project_id:
                    continue
                for each_block in each_rule['calcFormulaList']:
                    try:
                        if each_block['completeTagId'] == '':
                            continue
                        block_result_tag = f"{each_block['output_devices'][0]}${each_block['completeTagId']}"
                        rule_block_id = f"{each_rule['rule_engine_id']}.{each_block['block_id']}"
                        output_rule_block_map.update({block_result_tag: {"id": rule_block_id,
                                                                         "name": each_rule['ruleName']}})
                    except Exception as e:
                        logger.exception(f"Exception in get_rule_block_mapping {e}")
            return output_rule_block_map
        except Exception as e:
            logger.exception(f"{e}")
            raise Exception("Error fetching mapping details")

    def get_derived_tags(self):
        derived_tag_list = []
        try:
            aggregate_query = self.tags_aggregate.derived_tags_list
            res = self.tags_mongo.find_by_aggregate(aggregate_query)
            if res:
                for each in res:
                    derived_tag_list += each['derived_tag_list']
            return derived_tag_list
        except Exception as e:
            logger.exception(f"Exception in getting derived tag list {e}")
            raise Exception("Failed to fetch derived tags")

    def delete_rule_engine(self, input_data, delete_on_create=False):
        """ this function will delete rule data in mongo and REDIS"""
        try:
            rule_data = self.rules_mongo.find_one_rule(query={"rule_engine_id": input_data["rule_engine_id"]})
            if not rule_data:
                return {"status": "failed", "message": "Failed to delete rule"}
            created_on = rule_data.get('created_on', "")
            rule_type = rule_data.get('Selected_ruleType')
            for each_block in rule_data['calcFormulaList']:
                if rule_type and rule_type.lower().replace(" ", "") == RuleCommonKeys.SCHEDULE:
                    for each_tag, values in each_block.get('execute_on', dict()).items():
                        self.rules_redis.hdel(each_tag, input_data["rule_engine_id"])
                        for each_val in values:
                            self.rules_redis.hdel(f"{each_tag}_{each_val}", input_data["rule_engine_id"])
                    # if some tag is missed will delete here
                    for tag in [RuleCommonKeys.MINUTE, RuleCommonKeys.HOUR, RuleCommonKeys.DAY, RuleCommonKeys.MONTH,
                                RuleCommonKeys.SHIFT, RuleCommonKeys.WEEK, RuleCommonKeys.DAY_OF_WEEK,
                                RuleCommonKeys.YEAR]:
                        self.rules_redis.hdel(tag, input_data["rule_engine_id"])
                else:
                    self.remove_rule_from_redis(rule_data)
            if not delete_on_create:
                self.rules_mongo.delete_one_rule(query={"rule_engine_id": input_data["rule_engine_id"]})
            logger.debug("rule deleted successfully")
            return {"status": "success", "message": "rule deleted successfully", "created_on": created_on}
        except Exception as e:
            logger.exception("exceptions while deleting rule:" + str(e))
            return {"status": "failed", "message": "Unable to remove rule"}

    @staticmethod
    def make_json_to_push_into_redis(rule_block_data, input_data):
        """
        it will make json to push into redis
        """
        try:
            execute_on_tag_data = rule_block_data.get('execute_on_tag', [])
            execute_on_tag = [{k: v for k, v in d.items() if k != 'tagList'} for d in execute_on_tag_data]
            each_block_dict = {"rule_id": input_data["rule_engine_id"],
                               "rule": {"code": rule_block_data.get("code"),
                                        "parsedCode": rule_block_data.get("parsedCode")
                                        },
                               "result_tag_id": rule_block_data.get("completeTagId"),
                               "selectedTags": rule_block_data.get("selectedTags"),
                               "output_devices": rule_block_data.get("output_devices"),
                               "ruleName": input_data.get("ruleName"),
                               "process_on": input_data.get("processOn"),
                               "transformation_type": input_data["transformation_type"],
                               "output_type": rule_block_data.get("output_type"),
                               "mqtt": rule_block_data.get("mqtt"),
                               "kafka": rule_block_data.get("kafka"),
                               "rest": rule_block_data.get("rest"),
                               "data_store": rule_block_data.get("data_store"),
                               "output_tags": rule_block_data.get("output_tags"),
                               "block_id": rule_block_data.get("block_id"),
                               "disable": rule_block_data.get("disable", False),
                               "executeOnTag": rule_block_data.get("executeOnTag", False),
                               "rule_block_parent": [
                                   f"{input_data['rule_engine_id']}.{rule_block_data.get('block_id')}"],
                               "execute_on_tag": execute_on_tag
                               }
            return each_block_dict
        except Exception as e:
            logger.exception("unable to make json to push to redis:" + str(e))

    @staticmethod
    def get_execute_on(schedule_data):
        execute_on = {}
        try:
            if schedule_data['schedule_Type'].lower() == "onetime":
                execute_at = datetime.datetime.fromtimestamp(int(schedule_data['onetimeSchedule_DateAndTime']))
                if execute_at.year == datetime.datetime.now().year:
                    if execute_at.month == datetime.datetime.now().month:
                        if execute_at.day == datetime.datetime.now().day:
                            logger.debug("One Time Rule scheduled for current day")
                            cur_hr = datetime.datetime.now().hour
                            if execute_at.hour < cur_hr:
                                raise Exception("Rules can be configured only for future times")
                            elif execute_at.hour == cur_hr and execute_at.minute < datetime.datetime.now().minute:
                                raise Exception("Rules can be configured only for future times")
                        else:
                            execute_on = {f"{RuleCommonKeys.DAY}_{int(execute_at.day)}": ['*']}
                    else:
                        execute_on = {f"{RuleCommonKeys.MONTH}_{int(execute_at.month)}": ['*'],
                                      f"{RuleCommonKeys.DAY}_{int(execute_at.day)}": ['*']}
                else:
                    execute_on = {f"{RuleCommonKeys.YEAR}_{int(execute_at.year)}": ['*'],
                                  f"{RuleCommonKeys.MONTH}_{int(execute_at.month)}": ['*'],
                                  f"{RuleCommonKeys.DAY}_{int(execute_at.day)}": ['*']}
                execute_on[f"{RuleCommonKeys.MINUTE}_{int(execute_at.minute)}"] = ['*']
                execute_on[f"{RuleCommonKeys.HOUR}_{int(execute_at.hour)}"] = ['*']
            elif schedule_data['schedule_Type'] == "recurring":
                schedule_tag = schedule_data.get('scheduleInterval_Type')
                if schedule_tag:
                    custom_time = schedule_data['trigger_time'].split(":")
                    if int(custom_time[0]) == 0 and int(custom_time[1]) == 0:
                        execute_on[schedule_data.get('scheduleInterval_Type')] = list(
                            map(int, schedule_data[schedule_data.get('scheduleInterval_Type', [])]))
                    else:
                        execute_on[f"{RuleCommonKeys.HOUR}_{int(custom_time[0])}"] = ['*']
                        execute_on[f"{RuleCommonKeys.MINUTE}_{int(custom_time[1])}"] = ['*']
                        execute_on[schedule_data.get('scheduleInterval_Type')] = list(
                            map(int, schedule_data[schedule_data.get('scheduleInterval_Type', [])]))
            else:
                logger.debug("Inappropriate schedule type")
                raise Exception
            return execute_on
        except Exception as e:
            logger.exception(f"Exception while forming execute on {e}")
            raise Exception(e)

    def update_schedule_rule_to_redis(self, insert_block_json_list, execute_on, ruleid, schedule_type):
        """
            this function will insert schedule rule mapping into redis
        """
        try:
            conn_obj = self.rules_redis
            if schedule_type.lower() == 'recurring':
                keys_list = list(execute_on.keys())
                if len(keys_list) == 1:
                    key = keys_list[0]
                    conn_obj.hset(key, ruleid, json.dumps(insert_block_json_list))
                else:
                    for each_tag in execute_on:
                        if each_tag in [RuleCommonKeys.MONTH, RuleCommonKeys.DAY, RuleCommonKeys.DAY_OF_WEEK,
                                        RuleCommonKeys.WEEK]:
                            if each_tag == RuleCommonKeys.DAY and len(execute_on[each_tag]) == 31:
                                continue
                            else:
                                for val in execute_on[each_tag]:
                                    if val != '*':
                                        conn_obj.hset(f"{each_tag}_{val}", ruleid, json.dumps(insert_block_json_list))
                                    else:
                                        conn_obj.hset(each_tag, ruleid, json.dumps(insert_block_json_list))
                        else:
                            conn_obj.hset(each_tag, ruleid, json.dumps(insert_block_json_list))
            else:
                for each_tag in execute_on:
                    conn_obj.hset(each_tag, ruleid, json.dumps(insert_block_json_list))
        except Exception as e:
            logger.exception(f"Exception in schedule rule to redis {e}")
            raise Exception

    def update_realtime_rule_to_redis(self, rule_blocks):
        """
        this function will insert rule data into redis
        """
        try:
            conn_obj = self.rules_redis
            deleted_tag_list = []
            for json_to_redis in rule_blocks:
                for each_tag in json_to_redis.get('previousTags', []):
                    if each_tag not in deleted_tag_list:
                        deleted_tag_list.append(each_tag)
                        conn_obj.hdel(each_tag, json_to_redis["rule_id"])
            for json_to_redis in rule_blocks:
                if json_to_redis["transformation_type"] == "route":
                    tags_used_in_rule = json_to_redis["output_tags"]
                else:
                    tags_used_in_rule = json_to_redis["selectedTags"]
                for each_selected_tag in tags_used_in_rule:
                    if conn_obj.exists(each_selected_tag):
                        rules_available = conn_obj.hget(each_selected_tag, json_to_redis["rule_id"])
                        if not rules_available:
                            conn_obj.hset(each_selected_tag, json_to_redis["rule_id"], json.dumps([json_to_redis]))
                        else:
                            rules_available = json.loads(rules_available)
                            rules_available.append(json_to_redis)
                            conn_obj.hset(each_selected_tag, json_to_redis["rule_id"], json.dumps(rules_available))
                    else:
                        conn_obj.hset(each_selected_tag, json_to_redis["rule_id"], json.dumps([json_to_redis]))
        except Exception as e:
            logger.exception("unable to insert rule into redis:" + str(e))

    def duplicate_rule_name_check(self, input_data):
        """
        this will check whether that rule already exists while creations
        """
        exists = self.rules_mongo.find_one_rule(query={"ruleName": input_data["ruleName"]})
        if exists:
            return True
        else:
            return False

    def remove_rule_from_redis(self, rule_data):
        try:
            for each_block_data in rule_data["calcFormulaList"]:
                if rule_data["transformation_type"] == "route":
                    tags_used_in_rule = each_block_data["output_tags"]
                else:
                    tags_used_in_rule = each_block_data["selectedTags"]
                for each_selected_tag in tags_used_in_rule:
                    rules_available = self.rules_redis.hgetall(each_selected_tag)
                    if rule_data["rule_engine_id"] in rules_available.keys():
                        self.rules_redis.hdel(each_selected_tag, rule_data["rule_engine_id"])
        except Exception as e:
            logger.exception("Unable to clear REDIS cache:" + str(e))
