from typing import List, Optional, Dict

from scripts.constants import DatabaseNames, CollectionNames, TagsCollectionKeys
from scripts.logging import logger
from scripts.utils.mongo_util import MongoCollectionBaseClass


class Tags(MongoCollectionBaseClass):
    def __init__(self, mongo_client):
        super().__init__(mongo_client, database=DatabaseNames.ilens_configuration,
                         collection=CollectionNames.tags)

    @property
    def key_tag_category_id(self):
        return TagsCollectionKeys.KEY_TAG_CATEGORY_ID

    @property
    def key_tag_name(self):
        return TagsCollectionKeys.KEY_TAG_NAME

    @property
    def key_unit(self):
        return TagsCollectionKeys.KEY_UNIT

    @property
    def key_tag_type(self):
        return TagsCollectionKeys.KEY_TAG_TYPE

    @property
    def key_description(self):
        return TagsCollectionKeys.KEY_DESCRIPTION

    @property
    def key_id(self):
        return TagsCollectionKeys.KEY_ID

    @property
    def key_tag_group_id(self):
        return TagsCollectionKeys.KEY_TAG_GROUP_ID

    @property
    def key_default(self):
        return TagsCollectionKeys.KEY_DEFAULT

    @property
    def key_data_type(self):
        return TagsCollectionKeys.KEY_DATA_TYPE

    @property
    def key_system_tag_type(self):
        return TagsCollectionKeys.KEY_SYSTEM_TAG_TYPE

    @property
    def key_value_list(self):
        return TagsCollectionKeys.KEY_VALUE_LIST

    @property
    def key_product_encrypted(self):
        return TagsCollectionKeys.KEY_PRODUCT_ENCRYPTED

    def find_all_tags(self, filter_dict=None,
                      sort=None, skip=0, limit=None, **query):
        """
        The following function will give all tags for the given set of
        search parameters as keyword arguments
        :param filter_dict:
        :param sort:
        :param skip:
        :param limit:
        :param query:
        :return:
        """
        all_tags = self.find(filter_dict=filter_dict, sort=sort, skip=skip, limit=limit, query=query)
        if not all_tags:
            return list()
        return all_tags

    def find_tags_by_query(self, query, filter_dict=None):
        all_tags = self.find(query=query,filter_dict=filter_dict)
        if all_tags:
            return all_tags
        return list()

    def find_one_tag(self, filter_dict=None, **query):
        """
        The following function will give one tag for a given set of
        search parameters as keyword arguments
        :param filter_dict:
        :param query:
        :return:
        """
        tag = self.find_one(filter_dict=filter_dict, query=query)
        if tag:
            return tag
        else:
            return tag

    def insert_one_tag(self, data):
        """
        The following function will insert one tag in the
        tags collections
        :param data:
        :return:
        """
        return self.insert_one(data)

    def insert_many_tags(self, data):
        """
        The following function will insert many tags in the
        tags collection
        :param data:
        :return:
        """
        return self.insert_many(data)

    def update_one_tag(self, data, upsert=False, **query):
        """
        The following function will update one tag in
        tags collection based on the given query
        :param data:
        :param upsert:
        :param query:
        :return:
        """
        return self.update_one(data=data, upsert=upsert, query=query)

    def update_many_tags(self, data, query, upsert=False):
        """
        The following function will update many tags in
        tags collection based on the given query
        :param data:
        :param upsert:
        :param query:
        :return:
        """
        return self.update_many(data=data, upsert=upsert, query=query)

    def delete_many_tags(self, **query):
        """
        The following function will delete many tag in
        tags collection based on the given query
        :param query:
        :return:
        """
        return self.delete_many(query=query)

    def delete_one_tag(self, **query):
        """
        The following function will delete one tag in
        tags collection based on the given query
        :param query:
        :return:
        """
        if query:
            return self.delete_one(query=query)
        else:
            return False

    def distinct_tag(self, query_key):
        """
        Get a list of distinct values for `key` among all documents
        in the result set of this query.

        :param query_key:
        :return:
        """
        return self.distinct(query_key=query_key)

    def find_tags_with_list(self, tag_list):
        query = {self.key_tag_name: {"$in": tag_list}}
        filter_dict = dict(tag_name=1, _id=0)
        tags = self.find(query=query, filter_dict=filter_dict)
        if not tags:
            return list()
        return list(tags)

    def find_tags_by_ids(self, tag_category_id, tag_list):
        query = {"$or": [{self.key_tag_category_id: tag_category_id}, {self.key_id: {"$in": tag_list}}]}
        tags = self.find(query=query)
        if not tags:
            return list()
        return list(tags)

    def find_tags_by_aggregate(self, query):
        tags = self.aggregate(query)
        if not tags:
            return list()
        return list(tags)

    def update_many_tags_by_group(self, selected_tags, tag_group_id):
        data = {self.key_tag_group_id: tag_group_id}
        query = {self.key_id: {"$in": selected_tags}}
        self.update_many(query=query, data=data)

    def find_by_aggregate(self, query):
        tags = self.aggregate(query)
        if not tags:
            return list()
        return list(tags)

    def get_tag_name(self, tag_id):
        query = [
            {
                '$match': {
                    'id': tag_id
                }
            }, {
                '$project': {
                    '_id': 0,
                    'tag_name': '$tag_name',
                    'tag_id': '$tag_id'
                }
            }
        ]
        tags = self.aggregate(query)
        tags = [x for x in tags]
        if not tags:
            return ""
        return tags[0]["tag_name"]