
import os
import traceback
from datetime import datetime
import mysql.connector
import pandas as pd
from scripts.constants import app_configuration
from scripts.utils.logger import logger as log

DATA_DB_FILE = "manage.mdb"
META_DB_FILE = "NOZZLE.MDB"
META_TABLE = "Noztab1"

MS_ACCESS_CONFIG = "ms_access_config"

db_config = {
    "host": app_configuration.HOST,
    "port": app_configuration.PORT,
    "user": app_configuration.USER_NAME,
    "password": app_configuration.PASSWORD,
    "database": app_configuration.DB,
    "requestTimeout": 180000,
    "options": {
        "encrypt": True
    }
}


class MsAccessMount(object):

    def __init__(self):
        self.conn_flag = 0
        self.my_cursor = None
        self.create_connection()
        self.plant_dir_meta = None
        self.last_modified = {}
        self.plant_meta = []
        self.table = app_configuration.TABLE
        self.username = app_configuration.USERNAME

    def read_plant_meta(self):
        try:
            log.info("Reading plant meta")
            query = f"select * from {self.table} where user_name='{self.username}' ;"
            return pd.read_sql(query, con=self.mydb)
        except Exception as e:
            log.error(f"Error occurred while reading plant meta data : {str(e)}")
            raise Exception("Failed to read plant meta")

    def create_connection(self):
        try:
            log.info("*****Creating mysql connection*****")
            if not self.conn_flag:
                self.mydb = mysql.connector.connect(
                    host=db_config["host"],
                    port=db_config ["port"],
                    user=db_config["user"],
                    passwd=db_config["password"],
                    database=db_config["database"]
                )
                self.my_cursor = self.mydb.cursor()
                log.info("Created connection --->")
                self.conn_flag = 1
        except Exception as e:
            log.error("Error while connecting to mysql : {} !!!!!!!!!!!".format(str(e)))
            raise Exception("Failed to connect to mysql!!!!!!!!!!!!!!")

    def close_connection(self):
        try:
            log.info("*****Closing mysql connection*****")
            if self.conn_flag:
                self.my_cursor.close()
                self.mydb.close()
                self.conn_flag = 0
        except Exception as e:
            log.error("Error occurred while closing mysql connection : {} !!!!!!".format(str(e)))
            self.my_cursor.close()
            self.mydb.close()
            self.conn_flag = 0


    def __del__(self):
        self.my_cursor.close()
        self.mydb.close()

    def validate_path(self, path):
        try:
            # Check if the path is a directory and if the dir exists
            if not os.path.exists(path) and not os.path.isdir(path):
                log.error("File Path not found")
                self.update_failed_status(message="File Path not found")
                return

            # Check if the directory is readable
            if not os.access(path, os.R_OK):
                log.error("Directory is not readable")
                self.update_failed_status(message="Directory is not readable")
                return
            
            # Check if the required files with the given extension exist
            required_files = ["NOZZLE.MDB", "manage.1db", "manage.mdb"]
            missing_files = []
            for filename in required_files:
                lowercase_filename = filename.lower()
                file_path = os.path.join(path, filename)
                if not os.path.exists(file_path) and not os.path.exists(os.path.join(path, lowercase_filename)):
                    missing_files.append(filename)

            if missing_files:
                log.error("Some files are missing")
                self.update_failed_status(message="Some files are missing")
                return

            self.update_success_status(message="MS Access mounted successfully")
            return
        except Exception:
            log.error("Exception while validating file path", exc_info=True)

    def update_failed_status(self, message):
        try:
            update_ms_file_access_check = f"update {self.table} set status='failed', message={message} where user_name = '{self.username}'"
            resp = self.my_cursor.execute(update_ms_file_access_check)
            self.mydb.commit()
        except Exception as e:
            log.exception(f"Exception when updating status as Failed :{str(e)}")
            
    def update_success_status(self, message):
        try:
            log.info("Mounts validated successfully")
            update_ms_file_access_check = f"update {self.table} set status='success', message={message} where user_name = '{self.username}'"
            resp = self.my_cursor.execute(update_ms_file_access_check)
            self.mydb.commit()
        except Exception as e:
            log.exception(f"Exception when updating status as Success {str(e)}")
            
    def process_data(self):
        log.debug("Started Mountd path testing for Access DB")
        try:
            start_time = datetime.now()
            self.conn_flag = 0
            self.create_connection()
            log.info("Reading and refreshing the plant meta")
            self.plant_meta = self.read_plant_meta()
            self.plant_dir_meta = self.plant_meta.to_dict(orient='records')
            log.info(f"---------------> META ----------------> {self.plant_dir_meta}")
            if len(self.plant_dir_meta):
                self.validate_path(self.plant_dir_meta[0]["mounted_path"])
            end_time = datetime.now()
            diff = (end_time - start_time).total_seconds()
            log.debug("Time taken for execution : {} seconds".format(diff))
            self.close_connection()
            # break
        except Exception as ex:
            log.error("Error occurred while reading the data - main method --> %s", str(ex))
            traceback.print_exc()
            pass