import logging
import pandas as pd
from fastapi import APIRouter, UploadFile, File, Form

from scripts.config import Services
from scripts.constants.api import ApiEndpoints
from scripts.core.handlers.dashboard_handler import Dashboards
from scripts.core.handlers.widget_handler import Widgets
from scripts.utils.dashboard_utils import create_json

logger = logging.getLogger(Services.PROJECT_NAME)
router = APIRouter(prefix=ApiEndpoints.dashboards)
dashboard_handler = Dashboards()
widget_handler = Widgets()


@router.post(ApiEndpoints.dashboards + ApiEndpoints.create_new, tags=["1.create dashboard"])
async def create_dashboard(file: UploadFile = File(...), sheet_name: str = Form(), new_dashboard_name: str = Form(),
                           existing_dashboard_name: str = Form()):
    """

    :param existing_dashboard_name: name of existing dashboard which we need to take reference
    :param new_dashboard_name: name of new dashboard to be created
    :param file: Excel file with parameter details
    :param sheet_name: name of the sheet which we need to create dashboard
    :return: success or failure message
    """

    try:
        logger.info(f'service started in creating dashboard')
        with open(file.filename, "wb") as f:
            f.write(file.file.read())
        with pd.ExcelFile(file.filename, engine="openpyxl") as xls:
            parameters_df = pd.read_excel(xls, sheet_name=sheet_name)
            parameters_json = create_json(parameters_df)
        dashboard_name = {"dashboard_name": existing_dashboard_name}
        existing_dashboards = dashboard_handler.get_dashboard_details(dashboard_name)
        if existing_dashboards:
            existing_widget = widget_handler.get_widget_details(existing_dashboards)
            if existing_widget:
                dashboard_response, new_dashboard_name = dashboard_handler.create_new_dashboard(existing_dashboards,
                                                                                                new_dashboard_name)
                if dashboard_response:
                    new_dashboard_details = {
                        "dashboard_id": dashboard_response,
                        "name": new_dashboard_name
                    }
                    widget_response, missing_tags = widget_handler.create_new_widgets(existing_widget, parameters_json,
                                                                                      dashboard_response,
                                                                                      new_dashboard_details)
                    if widget_response or missing_tags:
                        return {
                            "widget": f'widget number:{widget_response} missing',
                            "tag": f'tag {missing_tags} not found'
                        }
            else:
                return {
                    "status": "failed",
                    "error": "error in getting widget details"
                }
        else:
            return {
                "status": "failed",
                "error": "error in getting dashboard details"
            }
    except Exception as e:
        logger.exception(f'error while creating dashboard {e}')
