import random
import time
from datetime import datetime, timezone

import requests
from apscheduler.triggers.cron import CronTrigger
from pytz import timezone as pytz_tz

from scripts.constants.app_constants import (
    APIJobConstants,
    CommonKeys,
    DatabaseConstants as Db,
)
from scripts.constants.db_connection import mongo_util
from scripts.core.engine.scheduler_engine import scheduler
from scripts.database.mongo.ilens_configuration.collections.schedule_metadata import ScheduleMetadataCollection
from scripts.errors.exceptions import ILensSchedulerError, ErrorMessages
from scripts.logging.logging import logger
from scripts.schemas.scheduler_schema import JobDetails, ScheduleJobRequest
from scripts.utils.common_utils import create_login_token

schedule_metadata_collection = ScheduleMetadataCollection()


class SchedulerHandler(object):
    def __init__(self):
        self.response_code_mapping = APIJobConstants.response_codes

    @staticmethod
    async def fetch_schedule_id(request_data):
        try:
            return schedule_metadata_collection.get_schedule_id(
                filters=request_data.filters
            )
        except Exception as e:
            logger.exception(e)
            raise

    @staticmethod
    def tz_diff(home_tz, away_tz):
        date = datetime.now()
        home = pytz_tz(home_tz)
        away = pytz_tz(away_tz)
        diff = (
                       home.localize(date) - away.localize(date).astimezone(home)
               ).seconds / 3600
        return int(diff), int((diff - int(diff)) * 60)

    @staticmethod
    async def fetch_all_schedule(request_data):
        try:
            return schedule_metadata_collection.get_all_schedules(
                filters=request_data.filters
            )
        except Exception as e:
            logger.exception(e)
            raise

    @staticmethod
    async def fetch_schedule_details(schedule_id):
        try:
            return schedule_metadata_collection.get_schedule_details(
                schedule_id=schedule_id
            )
        except Exception as e:
            logger.exception(e)
            raise

    @staticmethod
    async def fetch_schedule_table(project_id):
        try:
            filters = {"project_id": project_id}
            data = schedule_metadata_collection.get_all_schedules(filters=filters)
            header = [
                {"key": "job_type", "label": "Schedule Job Type"},
                {"key": "trigger_type", "label": "Trigger Type"},
            ]
            body = []
            for each in data:
                job_type = each.get("job_type")
                trigger_type = each["schedule_properties"].get("trigger_type")
                schedule_id = each.get("schedule_id")
                body.append(
                    {
                        "trigger_type": trigger_type,
                        "job_type": job_type,
                        "schedule_id": schedule_id,
                    }
                )

            return dict(header_content=header, body_content=body)
        except Exception as e:
            logger.exception(e)
            raise

    @staticmethod
    async def delete_schedule(schedule_id):
        try:
            schedule_metadata_collection.delete_schedule(schedule_id=schedule_id)
            if scheduler.get_job(job_id=schedule_id):
                scheduler.remove_job(job_id=schedule_id)
            return True
        except Exception as e:
            logger.exception(e)
            raise

    def create_scheduled_job(self, input_json: ScheduleJobRequest):
        try:
            save_msg = "Saving scheduler metadata to db"
            schedule_id = input_json.schedule_id
            logger.info(save_msg)
            if (
                    input_json.schedule_type is not None
                    and input_json.schedule_type == "advance"
            ):
                logger.info(save_msg)
                schedule_id = self.save_advanced_properties_to_mongo(
                    input_json, schedule_id
                )
                self.add_job_to_scheduler(input_json, schedule_id)
                return {"schedule_id": schedule_id}
            schedule_id = self.save_to_mongo(input_json, schedule_id=schedule_id)
            scheduler_type = input_json.scheduler_type
            if scheduler_type:
                logger.info(save_msg)
                self.add_job_to_scheduler(input_json, schedule_id)
            else:
                schedule_id = self.save_to_mongo(
                    input_json, schedule_id=schedule_id, saved_on_edge=False
                )
                logger.warning(
                    f"Scheduler with id '{schedule_id}' was not saved on edge"
                )
            return {"schedule_id": schedule_id}
        except Exception as e:
            raise ILensSchedulerError(f"{ErrorMessages.SCHEDULE_META_DATA}: {e}")

    def update_job_run_meta(self, **kwargs):
        try:
            db_data = {}
            create_run = kwargs.get("create_run", False)
            executor_properties = kwargs.get("executor", dict())
            input_properties = kwargs.get("input", dict())
            output_properties = kwargs.get("output", dict())
            run_id = kwargs.get("run_id", None)
            schedule_id = kwargs.get("schedule_id", None)
            status = kwargs.get("status", None)
            if create_run:
                run_id = self.generate_id(prefix="run_")
                db_data.update(
                    dict(
                        run_id=run_id,
                        schedule_id=schedule_id,
                        input=input_properties,
                        output=output_properties,
                        executor=executor_properties,
                        status=status,
                    )
                )
                mongo_util.insert_one(
                    database_name=Db.db_ilens_configuration,
                    collection_name=Db.collection_scheduled_job_runs,
                    data=db_data,
                )
            else:
                query = dict(run_id=run_id, schedule_id=schedule_id)
                db_data.update(dict(status=status, output=output_properties))
                mongo_util.update_one(
                    database_name=Db.db_ilens_configuration,
                    collection_name=Db.collection_scheduled_job_runs,
                    query=query,
                    data=db_data,
                )
            return run_id
        except Exception as e:
            raise ILensSchedulerError(f"{ErrorMessages.JOB_RUN_META_DATA}: {e}")

    def api_trigger(self, **kwargs):
        try:
            logger.info("Executing an API based job")
            request_url = kwargs.get("_api_url_", None)
            request_method = kwargs.get("_api_method_", None)
            request_payload = kwargs.get("_payload_", None)
            schedule_id = kwargs.get("_schedule_id_", None)
            project_id = kwargs.get("_project_id_", None)
            user_id = kwargs.get("_user_id_", None)
            try:
                run_id = self.update_job_run_meta(
                    schedule_id=schedule_id,
                    create_run=True,
                    status="submitted",
                    executor=dict(type="api", url=request_url, method=request_method),
                    input=dict(payload=request_payload),
                )
            except Exception as e:
                logger.critical(
                    f"iLens Scheduler faced a problem when generating a run id: {e}"
                )
                return False

            try:
                kwargs = dict(url=request_url)
                system_run_params = dict(_schedule_id_=schedule_id, _run_id_=run_id)
                kwargs.update(dict(params=system_run_params))
                if request_payload is not None:
                    request_payload.update(system_run_params)
                    kwargs.update(dict(json=request_payload))
                kwargs["cookies"] = {
                    "login-token": create_login_token(user_id=user_id, project_id=project_id), "project_id": project_id, "user_id": user_id,
                    "userId": user_id, "projectId": project_id
                }
                kwargs["headers"] = {
                    "login-token": create_login_token(user_id=user_id, project_id=project_id),
                    "projectId": project_id,
                }
                try:
                    self.update_job_run_meta(
                        status="running", run_id=run_id, schedule_id=schedule_id
                    )
                    res = getattr(requests, request_method)(**kwargs)
                except AttributeError:
                    self.update_job_run_meta(
                        status="failed",
                        run_id=run_id,
                        schedule_id=schedule_id,
                        output=dict(
                            error=f"Unsupported method '{request_method}' was given"
                        ),
                    )
                    logger.critical(
                        f"Unsupported method '{request_method}' was given. Failing the job."
                    )
                    raise ILensSchedulerError(
                        f"Unsupported method '{request_method}' was given. Failing the job."
                    )
                status_code = res.status_code
                res_message = res.json().get("message", None)
                res_error = res.json().get("error", None)
                status_code_mapping = self.response_code_mapping.get(
                    str(status_code), None
                )
                if status_code_mapping is None:
                    self.update_job_run_meta(
                        status="failed",
                        run_id=run_id,
                        schedule_id=schedule_id,
                        output=dict(
                            error=f"Job received a status code '{status_code}' from API response"
                        ),
                    )
                    logger.error(
                        f"Job received a status code '{status_code}' from API response"
                    )
                    return False
                else:
                    status = status_code_mapping.get("status", False)
                    message = status_code_mapping.get("message", None)
                    message = (
                        message.replace("#msg#", str(res_message))
                        .replace("#schedule_id#", str(schedule_id))
                        .replace("#url#", str(request_url))
                        .replace("#err#", str(res_error))
                    )
                    if status:
                        logger.info(message)
                        self.update_job_run_meta(
                            status="completed",
                            run_id=run_id,
                            schedule_id=schedule_id,
                            output=dict(message=message),
                        )
                        return True
                    else:
                        logger.warning(message)
                        self.update_job_run_meta(
                            status="failed",
                            run_id=run_id,
                            schedule_id=schedule_id,
                            output=dict(error=message),
                        )
                        return False
            except Exception as e:
                logger.error(
                    f"Scheduler faced an issue when running an API trigger job with run id {run_id}: {e}"
                )
                self.update_job_run_meta(
                    status="failed",
                    run_id=run_id,
                    schedule_id=schedule_id,
                    output=dict(error=f"Api Job failed with the following error: {e}"),
                )
                return False
        except Exception as e:
            logger.error(
                "Scheduler faced an unknown issue when running a job",
                exc_info=True,
            )

            return False

    @staticmethod
    def function_trigger():
        logger.info("Executing a function based job")

    def execute_job(self, job_details: JobDetails, schedule_id, project_id, user_id):
        try:
            job_type = job_details.execution_method
            job_properties = job_details.execution_properties
            if job_type == "api":
                logger.debug("Executing job of type api")
                trigger_args = []
                trigger_kwargs = {}
                api_url = job_properties.api_url
                api_method = job_properties.api_method
                payload = job_properties.payload
                trigger_kwargs.update(
                    dict(
                        _payload_=payload,
                        _api_url_=api_url,
                        _api_method_=api_method,
                        _schedule_id_=schedule_id,
                        _project_id_=project_id,
                        _user_id_=user_id,
                    )
                )
                return self.api_trigger, trigger_args, trigger_kwargs
            elif job_type == "func":
                logger.debug("Executing job of type function")
                raise NotImplementedError(
                    "Scheduler job type for 'func' is not implemented for this version of iLens"
                )

            else:
                raise ILensSchedulerError(
                    f"Job type '{job_type}' is not supported by iLens scheduler currently"
                )
        except Exception as e:
            raise ILensSchedulerError(
                f"{ErrorMessages.NECESSARY_JOB_DETAILS}: {e}"
            )

    def add_job_to_scheduler(self, input_data: ScheduleJobRequest, schedule_id):
        try:
            if (
                    input_data.schedule_type is not None
                    and input_data.schedule_type == "advance"
            ):
                cron_dict = self.get_advanced_cron_dict(
                    input_data.advanced_schedule_properties
                )
            else:
                cron_dict = self.get_cron_dict(input_data)
            logger.debug(f"Cron data: {cron_dict}")
            crontrigger = CronTrigger(**cron_dict)
            logger.info(f"Creating job for '{schedule_id}'")
            job_details = input_data.job_details
            project_id = input_data.project_id
            user_id = input_data.user_id
            trigger_obj, trigger_args, trigger_kwargs = self.execute_job(
                job_details=job_details,
                schedule_id=schedule_id,
                project_id=project_id,
                user_id=user_id,
            )
            logger.debug("Submitting job to scheduler framework")
            existing_job = scheduler.get_job(job_id=schedule_id)

            if existing_job is not None:
                scheduler.remove_job(job_id=schedule_id)

            if scheduler.state != 1:
                logger.info(
                    "Scheduler went to inactive state, Activating it to add new job"
                )
                scheduler.start()

            logger.info(f"Scheduler state: {scheduler.state}")

            scheduler_obj = scheduler.add_job(
                trigger_obj,
                crontrigger,
                id=schedule_id,
                args=trigger_args,
                kwargs=trigger_kwargs,
            )
            logger.info("Job added to scheduler")
            return scheduler_obj
        except Exception as e:
            raise ILensSchedulerError(f"{ErrorMessages.ERROR_ADD_JOB}: {e}")

    def save_advanced_properties_to_mongo(self, input_data, schedule_id=None):
        try:
            if schedule_id in [None, ""]:
                schedule_id = self.generate_id(prefix="job_")

            job_details = input_data.job_details.dict(skip_defaults=True)
            job_type = input_data.job_type
            scheduler_type = input_data.scheduler_type
            project_id = input_data.project_id
            logger.debug(f"Schedule ID: {schedule_id}")
            db_data = dict(
                schedule_id=schedule_id,
                advanced_schedule_properties=input_data.advanced_schedule_properties,
                job_details=job_details,
                job_type=job_type,
                scheduler_type=scheduler_type,
                project_id=project_id,
            )
            logger.debug(f"Pushing data to db: {db_data}")
            query = dict(schedule_id=schedule_id)
            schedule_data = mongo_util.find_one(
                database_name=Db.db_ilens_configuration,
                collection_name=Db.collection_scheduled_metadata,
                query=query,
            )
            if schedule_data is None:
                mongo_util.insert_one(
                    database_name=Db.db_ilens_configuration,
                    collection_name=Db.collection_scheduled_metadata,
                    data=db_data,
                )
            else:
                logger.debug("Updating an existing metadata record")
                mongo_util.update_one(
                    database_name=Db.db_ilens_configuration,
                    collection_name=Db.collection_scheduled_metadata,
                    query=query,
                    data=db_data,
                )
            logger.debug("Data stored to db successfully")
            return schedule_id
        except Exception as e:
            raise ILensSchedulerError(f"{ErrorMessages.SCHEDULE_META_DATA}: {e}")

    def save_to_mongo(
            self,
            input_data: ScheduleJobRequest,
            schedule_id=None,
            saved_on_edge=False,
    ):
        try:
            if schedule_id in [None, ""]:
                schedule_id = self.generate_id(prefix="job_")
            schedule_properties = input_data.schedule_properties.dict(
                skip_defaults=True
            )
            job_details = input_data.job_details.dict(skip_defaults=True)
            job_type = input_data.job_type
            scheduler_type = input_data.scheduler_type
            project_id = input_data.project_id
            hierarchy = input_data.hierarchy
            logger.debug(f"Schedule ID: {schedule_id}")
            db_data = dict(
                schedule_id=schedule_id,
                schedule_properties=schedule_properties,
                job_details=job_details,
                job_type=job_type,
                scheduler_type=scheduler_type,
                project_id=project_id,
                hierarchy=hierarchy,
                saved_on_edge=saved_on_edge,
            )
            logger.debug(f"Pushing data to db: {db_data}")
            query = dict(schedule_id=schedule_id)
            schedule_data = mongo_util.find_one(
                database_name=Db.db_ilens_configuration,
                collection_name=Db.collection_scheduled_metadata,
                query=query,
            )
            if schedule_data is None:
                mongo_util.insert_one(
                    database_name=Db.db_ilens_configuration,
                    collection_name=Db.collection_scheduled_metadata,
                    data=db_data,
                )
            else:
                logger.debug("Updating an existing metadata record")
                mongo_util.update_one(
                    database_name=Db.db_ilens_configuration,
                    collection_name=Db.collection_scheduled_metadata,
                    query=query,
                    data=db_data,
                )
            logger.debug("Data stored to db successfully")
            return schedule_id
        except Exception as e:
            raise ILensSchedulerError(
                f"{ErrorMessages.SCHEDULE_META_DATA}: {e}"
            )

    @staticmethod
    def generate_id(prefix):
        _id = prefix
        timestamp = time.time()
        rand = random.randint(1000, 9999)
        _id += str(int(timestamp)) + "_" + str(rand)
        return _id

    @staticmethod
    def days_hours_minutes(td):
        return td.days, td.seconds // 3600, (td.seconds // 60) % 60

    @staticmethod
    def system_timezone():
        return datetime.now(timezone.utc).astimezone().tzname()

    @staticmethod
    def get_advanced_cron_dict(scheduling_details):
        schedule_frequency = scheduling_details.get("schedule_frequency")
        cron_job_dict = {}
        recursion_count = scheduling_details.get(CommonKeys.KEY_RECURSION_COUNT)
        from_ts = scheduling_details.get("from_date")
        start_date = datetime.fromtimestamp(from_ts)
        cron_job_dict["start_date"] = start_date
        if "to_date" in scheduling_details:
            end_ts = scheduling_details.get("to_date")
            end_date = datetime.fromtimestamp(end_ts)
            cron_job_dict["end_date"] = end_date

        logger.info(f"System timezone is {SchedulerHandler.system_timezone()}")
        if "hours" in scheduling_details and "minutes" in scheduling_details:
            if SchedulerHandler.system_timezone() == "UTC":
                home_time = "{hr}:{min}".format(
                    hr=scheduling_details.get("hours"),
                    min=scheduling_details.get("minutes"),
                )
                diff_hours, diff_minutes = SchedulerHandler.tz_diff(
                    "UTC", scheduling_details.get(CommonKeys.KEY_TIMEZONE)
                )
                utc_time = "{hr}:{min}".format(hr=diff_hours, min=diff_minutes)
                format_ = "%H:%M"
                time = datetime.strptime(home_time, format_) - datetime.strptime(
                    utc_time, format_
                )
                days, hours, minutes = SchedulerHandler.days_hours_minutes(time)
            else:
                hours = scheduling_details.get("hours")
                minutes = scheduling_details.get("minutes")
        else:
            hours, minutes = SchedulerHandler.tz_diff(
                "UTC", scheduling_details.get(CommonKeys.KEY_TIMEZONE)
            )

        if schedule_frequency.lower() == CommonKeys.KEY_WEEKLY:
            cron_job_dict[CommonKeys.KEY_MINUTE] = str(minutes)
            cron_job_dict[CommonKeys.KEY_HOUR] = str(hours)
            cron_job_dict[CommonKeys.KEY_DAY] = "*"
            cron_job_dict[CommonKeys.KEY_MONTH] = "*"
            cron_job_dict[CommonKeys.KEY_DAY_OF_WEEK] = scheduling_details.get(
                CommonKeys.KEY_DAY_OF_WEEK, datetime.today().weekday()
            )
            cron_job_dict[CommonKeys.KEY_WEEK] = "*/" + str(recursion_count)
        elif schedule_frequency.lower() == CommonKeys.KEY_MONTHLY:
            cron_job_dict[CommonKeys.KEY_MINUTE] = str(minutes)
            cron_job_dict[CommonKeys.KEY_HOUR] = str(hours)
            cron_job_dict[CommonKeys.KEY_DAY] = str(
                scheduling_details.get(
                    CommonKeys.KEY_DAY_OF_MONTH, datetime.today().day
                )
            )
            cron_job_dict[CommonKeys.KEY_MONTH] = "*/" + str(recursion_count)
            cron_job_dict[CommonKeys.KEY_DAY_OF_WEEK] = "*"
        elif schedule_frequency.lower() == CommonKeys.KEY_YEARLY:
            cron_job_dict[CommonKeys.KEY_YEAR] = "*/" + str(recursion_count)
            cron_job_dict[CommonKeys.KEY_MINUTE] = str(minutes)
            cron_job_dict[CommonKeys.KEY_HOUR] = str(hours)
            cron_job_dict[CommonKeys.KEY_DAY] = str(
                scheduling_details.get(
                    CommonKeys.KEY_DAY_OF_MONTH, datetime.today().day
                )
            )
            cron_job_dict[CommonKeys.KEY_MONTH] = str(
                scheduling_details.get(
                    CommonKeys.KEY_MONTH_OF_YEAR, datetime.today().month
                )
            )
            cron_job_dict[CommonKeys.KEY_DAY_OF_WEEK] = "*"
        elif schedule_frequency.lower() == CommonKeys.KEY_DAILY:
            cron_job_dict[CommonKeys.KEY_YEAR] = "*"
            cron_job_dict[CommonKeys.KEY_MINUTE] = str(minutes)
            cron_job_dict[CommonKeys.KEY_HOUR] = str(hours)
            cron_job_dict[CommonKeys.KEY_DAY] = "*/" + str(recursion_count)
            cron_job_dict[CommonKeys.KEY_MONTH] = "*"
            cron_job_dict[CommonKeys.KEY_DAY_OF_WEEK] = "*"

        logger.info("cron_job_dict", cron_job_dict)

        return cron_job_dict

    @staticmethod
    def get_cron_dict(input_data: ScheduleJobRequest):
        cron_job_dict = {}
        rule_engine_data = input_data.schedule_properties
        trigger_type = rule_engine_data.trigger_type
        trigger_interval = rule_engine_data.trigger_interval
        trigger_datetime = rule_engine_data.interval_properties.trigger_date_time
        from_date = rule_engine_data.interval_properties.from_date
        to_date = rule_engine_data.interval_properties.to_date
        configure_daily_time_range = (
            rule_engine_data.interval_properties.configure_daily_time_range
        )
        daily_start_time = rule_engine_data.interval_properties.daily_start_time
        daily_end_time = rule_engine_data.interval_properties.daily_end_time
        run_on_day = rule_engine_data.interval_properties.run_on_day
        run_on_occurrence = rule_engine_data.interval_properties.run_on_occurrence
        trigger_time = rule_engine_data.interval_properties.trigger_time
        daily_selected_interval = rule_engine_data.interval_properties.selected_interval
        minute = rule_engine_data.interval_properties.minute
        hour = rule_engine_data.interval_properties.hour
        selected_week_days = rule_engine_data.interval_properties.selected_week_days
        selected_months = rule_engine_data.interval_properties.selected_months

        if trigger_type == "recurring":

            if configure_daily_time_range:
                daily_start_date = datetime.fromtimestamp(daily_start_time / 1000)
                cron_job_dict["start_date"] = daily_start_date

                schedule_hour = datetime.fromtimestamp(daily_start_time / 1000).hour
                schedule_minute = datetime.fromtimestamp(daily_start_time / 1000).minute
                cron_job_dict["hour"] = schedule_hour
                cron_job_dict["minute"] = schedule_minute

                daily_end_date = datetime.fromtimestamp(daily_end_time / 1000)
                cron_job_dict["end_date"] = daily_end_date

            if from_date and to_date:
                cron_job_dict["start_date"] = datetime.fromtimestamp(from_date / 1000)
                cron_job_dict["end_date"] = datetime.fromtimestamp(to_date / 1000)
            if run_on_occurrence is not None and run_on_day is not None:
                cron_job_dict["day"] = f"{run_on_occurrence} {run_on_day}"

            schedule_hour = 00
            schedule_minute = 00

            if trigger_time:
                schedule_hour = str(datetime.fromtimestamp(trigger_time / 1000).hour)
                schedule_minute = str(
                    datetime.fromtimestamp(trigger_time / 1000).minute
                )

            if trigger_interval == "every-n-minutes":
                count = int(minute)
                minutes_list = []
                i = 00
                while i < 59:
                    minutes_list.append(str(i))
                    i += count

                cron_job_dict["minute"] = ",".join(minutes_list)

            elif trigger_interval == "every-n-hourly":
                count = int(hour)
                hour_list = []
                i = 00
                while i < 24:
                    hour_list.append(str(i))
                    i += count

                cron_job_dict["hour"] = ",".join(hour_list)

                cron_job_dict["minute"] = "00"

            elif trigger_interval in ["daily", "weekly"]:
                if schedule_hour and schedule_minute:
                    cron_job_dict["hour"] = schedule_hour
                    cron_job_dict["minute"] = schedule_minute
                else:
                    cron_job_dict["hour"] = "00"
                    cron_job_dict["minute"] = "00"

                if trigger_interval == "daily" and daily_selected_interval:

                    if daily_selected_interval == "every_day":
                        cron_job_dict["day_of_week"] = "*"

                    elif daily_selected_interval == "week_days":
                        cron_job_dict["day_of_week"] = "mon-fri"

                    elif daily_selected_interval == "weekend":
                        cron_job_dict["day_of_week"] = "sat-sun"

                elif trigger_interval == "weekly" and selected_week_days:
                    cron_job_dict["day_of_week"] = ",".join(selected_week_days)
            elif trigger_interval == "monthly":
                date = datetime.fromtimestamp(from_date / 1000)
                # logger.trace('-----| Cron Job Dictionary before setting Monthly Meta rules: {}'.
                #             format(cron_job_dict))

                if selected_months:
                    cron_job_dict["month"] = ",".join(list(selected_months))
                else:
                    cron_job_dict["month"] = "*"

                if not cron_job_dict["day"]:
                    cron_job_dict["day"] = str(date.day)
                if schedule_hour and schedule_minute:
                    cron_job_dict["hour"] = schedule_hour
                    cron_job_dict["minute"] = schedule_minute
                else:
                    cron_job_dict["hour"] = "00"
                    cron_job_dict["minute"] = "00"

        elif trigger_type == "onetime" and trigger_datetime:
            start_date = datetime.fromtimestamp(trigger_datetime / 1000)
            cron_job_dict["year"] = str(start_date.year)
            cron_job_dict["month"] = str(start_date.month)
            cron_job_dict["day"] = str(start_date.day)
            cron_job_dict["hour"] = str(start_date.hour)
            cron_job_dict["minute"] = str(start_date.minute)
        logger.debug(f"cron_dict: {cron_job_dict}")
        return cron_job_dict
