from fastapi import APIRouter
from pydantic import ValidationError

from scripts.constants.app_constants import APIEndpoints, APIJobConstants
from scripts.core.handlers.scheduler_handler import SchedulerHandler
from scripts.logging.logging import logger
from scripts.schemas.response_models import DefaultFailureResponse, DefaultResponse
from scripts.schemas.scheduler_schema import (
    ScheduledIDFetchRequest,
    ScheduleJobRequest,
    SchedulesFetchRequest,
)

scheduler_handler = SchedulerHandler()

scheduler_router = APIRouter(prefix=APIEndpoints.scheduler_base)

health_check = APIRouter()


@health_check.get("/api/iLens-schedular/healthcheck")
async def ping():
    return {"status": 200}


@scheduler_router.post(APIEndpoints.schedule, tags=["scheduler"])
async def schedule(request_data: ScheduleJobRequest):
    try:
        response_json = scheduler_handler.create_scheduled_job(request_data)
        return DefaultResponse(status="success", data=response_json)
    except ValidationError as e:
        logger.error(f"Request data model validation failed: {e.json()}")
        return DefaultFailureResponse(
            status="failed",
            message=APIJobConstants.validation_failed,
            error=e.json(),
        )
    except Exception as e:
        logger.error(f"Faced a problem when scheduling the job: {e}")
        return DefaultFailureResponse(
            status="failed",
            message=APIJobConstants.validation_failed,
            error=e,
        )


@scheduler_router.post(APIEndpoints.fetch_schedule_id, tags=["scheduler"])
async def fetch_schedule_id(request_data: ScheduledIDFetchRequest):
    try:
        response_json = await scheduler_handler.fetch_schedule_id(request_data)

        return DefaultResponse(status="success", data=response_json)
    except ValidationError as e:
        logger.error(f"Request data model validation failed: {e.json()}")
        return DefaultFailureResponse(
            status="failed",
            message=APIJobConstants.validation_failed,
            error=e.json(),
        )
    except Exception as e:
        logger.error(f"Faced a problem when scheduling the job: {e}")
        return DefaultFailureResponse(
            status="failed",
            message=APIJobConstants.scheduling_failure,
            error=e,
        )


@scheduler_router.post(APIEndpoints.fetch_schedules, tags=["scheduler"])
async def fetch_schedules(request_data: SchedulesFetchRequest):
    try:
        response_json = await scheduler_handler.fetch_all_schedule(request_data)

        return DefaultResponse(status="success", data=response_json)
    except ValidationError as e:
        logger.error(f"Request data model validation failed: {e.json()}")
        return DefaultFailureResponse(
            status="failed",
            message=APIJobConstants.validation_failed,
            error=e.json(),
        )
    except Exception as e:
        logger.error(f"Faced a problem when scheduling the job: {e}")
        return DefaultFailureResponse(
            status="failed",
            message=APIJobConstants.scheduling_failure,
            error=e,
        )


@scheduler_router.get(APIEndpoints.fetch_schedule_details, tags=["scheduler"])
async def fetch_schedule_details(schedule_id: str):
    try:
        response_json = await scheduler_handler.fetch_schedule_details(
            schedule_id=schedule_id
        )
        return DefaultResponse(status="success", data=response_json)
    except ValidationError as e:
        logger.error(f"Request data model validation failed: {e.json()}")
        return DefaultFailureResponse(
            status="failed",
            message=APIJobConstants.validation_failed,
            error=e.json(),
        )
    except Exception as e:
        logger.error(f"Faced a problem when scheduling the job: {e}")
        return DefaultFailureResponse(
            status="failed",
            message=APIJobConstants.scheduling_failure,
            error=e,
        )


@scheduler_router.get(APIEndpoints.fetch_schedule_details_table, tags=["scheduler"])
async def fetch_schedule_table(project_id: str):
    try:
        response_json = await scheduler_handler.fetch_schedule_table(
            project_id=project_id
        )
        return DefaultResponse(
            status="success",
            data=response_json,
            message="Data Fetched Successfully",
        )
    except ValidationError as e:
        logger.error(f"Request data model validation failed: {e.json()}")
        return DefaultFailureResponse(
            status="failed",
            message=APIJobConstants.validation_failed,
            error=e.json(),
        )
    except Exception as e:
        logger.error(f"Faced a problem when scheduling the job: {e}")
        return DefaultFailureResponse(
            status="failed",
            message=APIJobConstants.scheduling_failure,
            error=e,
        )


@scheduler_router.get(APIEndpoints.delete_schedule, tags=["scheduler"])
async def fetch_schedule_details(schedule_id: str):
    try:
        response_json = await scheduler_handler.delete_schedule(schedule_id=schedule_id)
        return DefaultResponse(
            status="success",
            data=response_json,
            message="Job Deleted Successfully",
        )
    except ValidationError as e:
        logger.error(f"Request data model validation failed: {e.json()}")
        return DefaultFailureResponse(
            status="failed",
            message=APIJobConstants.validation_failed,
            error=e.json(),
        )
    except Exception as e:
        logger.error(f"Faced a problem when scheduling the job: {e}")
        return DefaultFailureResponse(
            status="failed",
            message=APIJobConstants.scheduling_failure,
            error=e,
        )
