if __name__ == '__main__':
    from dotenv import load_dotenv

    load_dotenv()
import os

import uvicorn
from fastapi import FastAPI, Depends
from fastapi.middleware.cors import CORSMiddleware
from jwt_signature_validator.encoded_payload import (EncodedPayloadSignatureMiddleware)

from scripts.constants.app_configuration import Service
from scripts.constants.app_constants import Secrets
from scripts.core.engine.scheduler_engine import scheduler
from scripts.logging.logging import logger
from scripts.services.scheduler import scheduler_router, health_check
from scripts.utils.security_utils.decorators import CookieAuthentication

secure_access = os.environ.get("SECURE_ACCESS", default=False)
protected_hosts = os.environ.get("PROTECTED_HOSTS", "").split(",")
verify_signature = os.environ.get("verify_signature", default=False)
auth = CookieAuthentication()

app = FastAPI(
    title="Scheduler Microservice",
    version="1.0",
    description="Scheduler App",
    openapi_url=os.environ.get("SW_OPENAPI_URL", default="/openapi.json"),
    docs_url=os.environ.get("SW_DOCS_URL"),
    redoc_url=None
)

if os.environ.get("ENABLE_CORS") in (True, 'true', 'True') and os.environ.get("CORS_URLS"):
    origins_list = os.environ.get("CORS_URLS", default="")
    origins_list = origins_list.split(',') if origins_list else ["*"]


if verify_signature:
    app.add_middleware(
        EncodedPayloadSignatureMiddleware,
        jwt_secret=Secrets.signature_key,
        jwt_algorithms=Secrets.signature_key_alg,
        protect_hosts=protected_hosts,
    )
    
if secure_access in [True, 'true', 'True']:
    app.include_router(scheduler_router, dependencies=[Depends(auth)])
else:
    app.include_router(scheduler_router)
app.include_router(health_check)

service_obj = Service()

if __name__ == "__main__":
    try:
        logger.info("Starting the scheduler framework")
        scheduler.start()
        logger.info("Scheduler framework started successfully")
        uvicorn.run(app, host=service_obj.host, port=service_obj.port)
    except (KeyboardInterrupt, SystemExit):
        scheduler.shutdown()
        raise
