import json
from enum import Enum
from typing import Dict, Optional, List

from pydantic import BaseModel, root_validator


class NodeActionOptions(str, Enum):
    add = "add"
    delete = "delete"
    update = "update"


class Edges(BaseModel):
    node_id: Optional[str]
    action: NodeActionOptions
    rel_name: str
    new_rel_name: Optional[str]
    bind_to: str

    class Config:
        allow_population_by_field_name = True


class InputRequestSchema(BaseModel):
    node_id: str
    node_name: str
    action: NodeActionOptions
    project_id: str
    node_type: Optional[str] = "Events"
    properties: Optional[Dict] = {}
    node_properties: Optional[str]
    edges: Optional[List[Edges]]

    class Config:
        allow_population_by_field_name = True

    @root_validator
    def date_range_validator(cls, values):
        if values['properties']:
            values['node_properties'] = json.dumps(values['properties'])
        return values


class GraphData(BaseModel):
    __root__: Dict[str, InputRequestSchema]

    class Config:
        allow_population_by_field_name = True
        schema_extra = {
            "example": {
                "Node1": {
                    "id": "BSCH270120022",
                    "action": "add",
                    "name": "Event Name",
                    "properties": {
                        "name": "Event 1",
                        "external_data_source": "mongo",
                        "external_data_id": "101",
                        "event_info": "847263",
                        "received_date": 643308200,
                        "completed_date": 1643394600
                    },
                    "edges": [],
                },
                "Node2": {
                    "id": "BSCH270120023",
                    "action": "add",
                    "name": "Event Name",
                    "properties": {
                        "name": "Event 2",
                        "external_data_source": "mongo",
                        "external_data_id": "101",
                        "event_info": "847263",
                        "received_date": 643308200,
                        "completed_date": 1643394600
                    },
                    "edges": [{
                        "action": "add",
                        "rel_name": "causes",
                        "bind_to": "Node1",
                        "bind_id": "BSCH270120022"
                    }]
                }
            }
        }


class GetNodeInfo(BaseModel):
    project_id: str
    id: str


class ResponseModelSchema(BaseModel):
    series_data: Optional[dict]
