from enum import Enum
from typing import Dict, Optional, List

from pydantic import BaseModel, Field


class NodeActionOptions(str, Enum):
    add = "add"
    delete = "delete"
    update = "update"


class Edges(BaseModel):
    node_id: Optional[str]
    action: NodeActionOptions
    rel_name: str
    bind_to: str
    bind_id: str

    class Config:
        allow_population_by_field_name = True


class InputRequestSchema(BaseModel):
    node_id: str
    node_name: str
    action: NodeActionOptions
    node_properties: Optional[Dict] = {}
    edges: Optional[List[Edges]]

    class Config:
        allow_population_by_field_name = True


class NodeSchemaValidation(BaseModel):
    node_id: str = Field(alias="node_id")
    name: str = Field(alias="node_name")
    properties: Optional[Dict] = Field(alias="node_properties", default={})
    label: str

    class Config:
        allow_population_by_field_name = True


class GraphData(BaseModel):
    __root__: Dict[str, InputRequestSchema]

    class Config:
        allow_population_by_field_name = True
        schema_extra = {
            "example": {
                "Node1": {
                    "id": "BSCH270120022",
                    "action": "add",
                    "name": "Event Name",
                    "properties": {
                        "name": "Event 1",
                        "external_data_source": "mongo",
                        "external_data_id": "101",
                        "event_info": "847263",
                        "received_date": 643308200,
                        "completed_date": 1643394600
                    },
                    "edges": [],
                },
                "Node2": {
                    "id": "BSCH270120023",
                    "action": "add",
                    "name": "Event Name",
                    "properties": {
                        "name": "Event 2",
                        "external_data_source": "mongo",
                        "external_data_id": "101",
                        "event_info": "847263",
                        "received_date": 643308200,
                        "completed_date": 1643394600
                    },
                    "edges": [{
                        "action": "add",
                        "rel_name": "causes",
                        "bind_to": "Node1",
                        "bind_id": "BSCH270120022"
                    }]
                }
            }
        }
