from typing import List

from gqlalchemy import GQLAlchemyError

from scripts.db.graphdb.neo4j import Neo4jHandler
from scripts.db.models import RelationShipMapper, NodePropertiesSchema, NodeFetchSchema
from scripts.logging import logger


class GraphUtility:

    def __init__(self, db):
        self.db = db

    def save_single_node(self, node_data: NodePropertiesSchema):
        try:
            if node_info := self.load_node_data_from_graph(node_data):
                node_data._id = node_info._id
                return self.db.save_node_with_id(node_data)
            else:
                return self.db.save_node(node_data)
        except Exception as e:
            logger.exception(f'Exception occurred while creating node for the graph {e.args}')
            raise

    def save_multiple_nodes(self, node_data: List[NodePropertiesSchema]):
        try:
            res = self.db.save_nodes(nodes=node_data)
        except Exception as e:
            logger.exception(f'Exception occurred while creating node for the graph {e.args}')
            raise

    def save_relationship(self, rel_data: RelationShipMapper, new_rel_name=None):
        try:
            if not (rel_info := self.load_relationship_data_from_graph(rel_data)):
                return self.db.save_relationship(rel_data)
            neo4j_handler = Neo4jHandler(self.db)
            neo4j_handler.delete_relationship_by_id(rel_info)
            if new_rel_name:
                rel_data._type = new_rel_name
            return self.db.save_relationship(rel_data)
        except Exception as e:
            logger.exception(f'Exception occurred while creating node for the graph {e.args}')
            raise

    def save_multiple_relationship(self, rel_data: List[RelationShipMapper]):
        try:
            res = self.db.save_relationships(relationships=rel_data)
        except Exception as e:
            logger.exception(f'Exception occurred while creating node for the graph {e.args}')
            raise

    def load_node_data_from_graph(self, node_data: NodePropertiesSchema):
        try:
            return self.db.load_node(node_data)
        except GQLAlchemyError as e:
            logger.debug(f'{e.args}')
            return None
        except Exception as e:
            logger.exception(f'Exception Occurred while fetching the node details by {node_data.id} --> {e.args}')
            raise

    def load_relationship_data_from_graph(self, rel_data: RelationShipMapper):
        try:
            return self.db.load_relationship(rel_data)
        except GQLAlchemyError as e:
            logger.debug(f'{e.args}')
            return None
        except Exception as e:
            logger.exception(f'Exception Occurred while fetching the relation details  -> {e.args}')
            raise
