import os
import shutil
import sys
from configparser import BasicInterpolation, ConfigParser


class EnvInterpolation(BasicInterpolation):
    """
    Interpolation which expands environment variables in values.
    """

    def before_get(self, parser, section, option, value, defaults):
        value = super().before_get(parser, section, option, value, defaults)

        if not os.path.expandvars(value).startswith('$'):
            return os.path.expandvars(value)
        else:
            return


try:
    config = ConfigParser(interpolation=EnvInterpolation())
    config.read("conf/application.conf")
except Exception as e:
    print(f"Error while loading the config: {e}")
    print("Failed to Load Configuration. Exiting!!!")
    sys.stdout.flush()
    sys.exit()


class Service:
    MODULE_NAME = config["MODULE"]["name"]
    PORT = config["SERVICE"]["port"]
    HOST = config["SERVICE"]["host"]
    verify_signature = os.environ.get("VERIFY_SIGNATURE", False)
    protected_hosts = os.environ.get("PROTECTED_HOSTS", "").split(",")


class DBConf:
    GRAPH_HOST = config.get('GRAPH_DB', 'GRAPH_HOST')
    GRAPH_PORT = config.get('GRAPH_DB', 'GRAPH_PORT')
    GRAPH_USERNAME = config.get('GRAPH_DB', 'GRAPH_USERNAME')
    GRAPH_PASSWORD = config.get('GRAPH_DB', 'GRAPH_PASSWORD')
    GRAPH_DB_TYPE = config.get('GRAPH_DB', 'DB_TYPE')
    if not GRAPH_DB_TYPE:
        GRAPH_DB_TYPE = "neo4j"


class Logging:
    level = config.get("LOGGING", "level", fallback="DEBUG")
    level = level or "INFO"
    tb_flag = config.getboolean("LOGGING", "traceback", fallback=True)
    tb_flag = tb_flag if tb_flag is not None else True


class RedisConfig(object):
    uri = config.get("REDIS", "uri")
    login_db = config["REDIS"]["login_db"]
    project_tags_db = config.getint("REDIS", "project_tags_db")
    user_role_permissions = config.getint("REDIS", "user_role_permissions")


class KeyPath(object):
    keys_path = config['DIRECTORY']['keys_path']
    if not os.path.isfile(os.path.join(keys_path, "public")) or not os.path.isfile(
            os.path.join(keys_path, "private")):
        if not os.path.exists(keys_path):
            os.makedirs(keys_path)
        shutil.copy(os.path.join("assets", "keys", "public"), os.path.join(keys_path, "public"))
        shutil.copy(os.path.join("assets", "keys", "private"), os.path.join(keys_path, "private"))
    public = os.path.join(keys_path, "public")
    private = os.path.join(keys_path, "private")
