import os

from fastapi import FastAPI, Depends
from fastapi.middleware.cors import CORSMiddleware
from jwt_signature_validator.encoded_payload import (
    EncodedPayloadSignatureMiddleware as SignatureVerificationMiddleware
)

from scripts.config import Service
from scripts.constants import Secrets
from scripts.services import service_router
from scripts.utils.security_utils.decorators import CookieAuthentication

auth = CookieAuthentication()
secure_access = os.environ.get("SECURE_ACCESS", default=False)

app = FastAPI(
    title="GraphDB Management",
    version="1.0.0",
    description="Graph Management App",
    openapi_url=os.environ.get("SW_OPENAPI_URL"),
    docs_url=os.environ.get("SW_DOCS_URL"),
    redoc_url=None,
    root_path="/rel_mnt"
)

if Service.verify_signature in [True, 'True', 'true']:
    app.add_middleware(
        SignatureVerificationMiddleware,
        jwt_secret=Secrets.signature_key,
        jwt_algorithms=Secrets.signature_key_alg,
        protect_hosts=Service.protected_hosts,
    )

origins_list = os.environ.get("CORS_URLS", default="")
origins_list = origins_list.split(',') if origins_list else ["*"]
app.add_middleware(
    CORSMiddleware,
    allow_origins=origins_list,
    allow_credentials=True,
    allow_methods=["GET", "POST", "DELETE", "PUT"],
    allow_headers=["*"],
)


@app.get(f"/api/{Service.MODULE_NAME}/healthcheck")
def ping():
    return {"status": 200}


auth_enabled = [Depends(auth)] if secure_access in [True, 'true', 'True'] else []
app.include_router(service_router, dependencies=auth_enabled)
