import requests
from pandas import DataFrame
from datetime import datetime

from scripts.common.config_parser import *
from scripts.common.logsetup import logger
from scripts.utils import get_aggregation_query
from scripts.common.constants import KariosConstants, ComponentExceptions


def get_data(query):
    logger.info("Parsing user requests")

    #  ------------------------ Validating Kairos URL Key ------------------------ #
    if KariosConstants.KAIROS_URL_KEY in config.keys():
        kairosdb_server = config[KariosConstants.KAIROS_URL_KEY]
    else:
        raise Exception(ComponentExceptions.INVALID_KAIROS_URL_EXCEPTION)

    #  ---------------------- Validating Tag Hierarchy Key ----------------------- #
    if config[KariosConstants.TAG_HIERARCHY_KEY] is not None and len(config[KariosConstants.TAG_HIERARCHY_KEY]) > 0:
        pass
    else:
        raise Exception(ComponentExceptions.INVALID_TAG_HIERARCHY_EXCEPTION)

    #  ------------------------ Validating Metric Name Key ----------------------- #
    if KariosConstants.METRIC_NAME_KEY in config.keys():
        query[KariosConstants.METRICS_KEY][0][KariosConstants.NAME_KEY] = config[KariosConstants.METRIC_NAME_KEY]
    else:
        raise Exception(ComponentExceptions.INVALID_METRIC_NAME_EXCEPTION)

    #  ------------------------ Parsing Start Time ------------------------------- #
    """
    If start absolute Key is present, it will take precedence over start relative key.
    For start relative key is there, proper unit has to be provided.
    """
    if config[KariosConstants.START_ABSOLUTE_KEY] != '':
        start_time = config[KariosConstants.START_ABSOLUTE_KEY]
        try:
            start_timestamp = datetime.strptime(start_time, KariosConstants.DATETIME_FORMAT)
            timestamp_start = datetime.timestamp(start_timestamp)
            query[KariosConstants.START_ABSOLUTE_KEY] = int(timestamp_start) * 1000
            query.pop(KariosConstants.START_RELATIVE_KEY)
        except ValueError as e:
            raise Exception(f"{ComponentExceptions.INVALID_START_TIME_FORMAT_EXCEPTION} - {str(e)}")

    elif config[KariosConstants.START_RELATIVE_KEY] != '':
        if int(config[KariosConstants.START_RELATIVE_KEY]) > 0:
            query[KariosConstants.START_RELATIVE_KEY][KariosConstants.VALUE_KEY] = config[KariosConstants.
                START_RELATIVE_KEY]
            query[KariosConstants.START_RELATIVE_KEY][KariosConstants.UNIT_KEY] = config[KariosConstants.
                START_RELATIVE_UNIT_KEY].lower()
            query.pop(KariosConstants.START_ABSOLUTE_KEY)
        else:
            raise Exception(ComponentExceptions.INVALID_START_TIME_VALUE_EXCEPTION)
    else:
        raise Exception(ComponentExceptions.MISSING_START_TIME_VALUE_EXCEPTION)

    #  ------------------------ Parsing End Time ------------------------------- #
    """
    If end absolute Key is present, it will take precedence over end relative key.
    For end relative key is there, proper unit has to be provided.
    If neither of them are there, then data will be queried for eternity.
    """
    if config[KariosConstants.END_ABSOLUTE_KEY] != '':
        end_time = config[KariosConstants.END_ABSOLUTE_KEY]
        try:
            end_timestamp = datetime.strptime(end_time, KariosConstants.DATETIME_FORMAT)
            timestamp_end = datetime.timestamp(end_timestamp)
            query[KariosConstants.END_ABSOLUTE_KEY] = int(timestamp_end) * 1000
            query.pop(KariosConstants.END_RELATIVE_KEY)
        except ValueError as e:
            raise Exception(f"{ComponentExceptions.INVALID_END_TIME_FORMAT_EXCEPTION} - {str(e)}")

    elif config[KariosConstants.END_RELATIVE_KEY] != '':
        if int(config[KariosConstants.END_RELATIVE_KEY]) > 0:
            query[KariosConstants.END_RELATIVE_KEY][KariosConstants.VALUE_KEY] = config[KariosConstants.
                END_RELATIVE_KEY][KariosConstants.VALUE_KEY]
            query[KariosConstants.END_RELATIVE_KEY][KariosConstants.UNIT_KEY] = config[KariosConstants.
                END_RELATIVE_UNIT_KEY].lower()
            query.pop(KariosConstants.END_ABSOLUTE_KEY)
        else:
            raise Exception(ComponentExceptions.INVALID_START_END_VALUE_EXCEPTION)
    else:
        query.pop(KariosConstants.END_ABSOLUTE_KEY)
        query.pop(KariosConstants.END_RELATIVE_KEY)

    #  ------------------------ Parsing Aggregation Key ------------------------------- #
    if config.get(KariosConstants.AGGREGATION_OPS) != "None" or config.get(KariosConstants.AGGREGATION_OPS) is not None:
        query[KariosConstants.METRICS_KEY][0][KariosConstants.AGGREGATORS_KEY].append(get_aggregation_query(config))
    else:
        query[KariosConstants.METRICS_KEY][0].pop(KariosConstants.AGGREGATORS_KEY)

    #  ----------------- Parsing Group BY Key - [TO BE IMPLEMENTED] ------------------- #
    if KariosConstants.GROUPBY_KEY not in config.keys():
        query[KariosConstants.METRICS_KEY][0].pop(KariosConstants.GROUPBY_KEY)
    if KariosConstants.TAG_HIERARCHY_KEY in config.keys():
        query[KariosConstants.METRICS_KEY][0][KariosConstants.TAGS_KEY][KariosConstants.C3_KEY] = \
            [list(config[KariosConstants.TAG_HIERARCHY_KEY].values())[0]]
    else:
        raise Exception(ComponentExceptions.MISSING_TAG_HIERARCHY_EXCEPTION)

    #  --------------------------- Querying KairosDB --------------------------------- #
    logger.info("Querying Karios DB...")
    logger.debug(KariosConstants.LOG_VAR_MESSAGE.format("Query", json.dumps(query, indent=1)))
    response = requests.post(kairosdb_server + KariosConstants.KARIOS_API, data=json.dumps(query))
    if response.status_code == KariosConstants.REQUEST_SUCCESS_CODE:
        logger.info("Receiving data...")
        data = response.json()[KariosConstants.QUERIES_KEY][0][KariosConstants.RESULTS_KEY][0] \
            [KariosConstants.VALUES_KEY]
        df = DataFrame.from_dict(data)
        if len(df) > 0:
            df.columns = [KariosConstants.TIMESTAMP_COLUMN_NAME, list(config[KariosConstants.TAG_HIERARCHY_KEY].keys())
            [0]]
        else:
            logger.warn("The dataframe is empty!")
            df = DataFrame(columns=[KariosConstants.TIMESTAMP_COLUMN_NAME, list(config[KariosConstants.
                                                                                TAG_HIERARCHY_KEY].keys())[0]])
        logger.debug("Dataframe --> \n{}".format(df.head()))

        return df
    else:
        raise Exception(response.json()[KariosConstants.ERRORS_KEY])


if __name__ == '__main__':
    logger.debug(KariosConstants.LOG_VAR_MESSAGE.format("COMPONENT CONFIG", json.dumps(config, indent=1)))
    data = get_data(KariosConstants.QUERY)
    data.to_csv(os.path.join(config['shared_volume'], 'data.csv'), index=False)
    logger.info("Data successfully written to --> {}".format(os.path.join(config['shared_volume'], 'data.csv')))
