# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License.

from botbuilder.dialogs import (
    ComponentDialog,
    WaterfallDialog,
    WaterfallStepContext,
    DialogTurnResult,
)
from botbuilder.dialogs.prompts import TextPrompt, PromptOptions
from botbuilder.core import MessageFactory, TurnContext
from botbuilder.schema import InputHints
from user_profile import UserProfile
from ilens_luis import IlensLuis
from helpers.luis_helper import LuisHelper, Intent
from botbuilder.schema import ChannelAccount, CardAction, ActionTypes, SuggestedActions
from botbuilder.dialogs.prompts import ConfirmPrompt, ChoicePrompt, PromptOptions
from botbuilder.dialogs.choices import Choice
from .task_details import task_details
from .qna import faq
from botbuilder.core import ActivityHandler, ConversationState, UserState, TurnContext


class MainDialog(ComponentDialog):
    def __init__(
            self, luis_recognizer: IlensLuis,user_state: UserState, task_details: task_details, faq: faq
    ):
        super(MainDialog, self).__init__(MainDialog.__name__)
        self.task_details_id = task_details.id
        self.faq_id = faq.id
        self.add_dialog(task_details)
        self.add_dialog(faq)
        self.add_dialog(ChoicePrompt(ChoicePrompt.__name__))
        self.add_dialog(ConfirmPrompt(ConfirmPrompt.__name__))
        self._luis_recognizer = luis_recognizer

        self.add_dialog(TextPrompt(TextPrompt.__name__))
        self.add_dialog(
            WaterfallDialog(
                "WFDialog", [self.welcome_step, self.options_step, self.final_step]
            )
        )
        self.user_state = user_state
        self.user_profile_accessor = self.user_state.create_property("UserProfile")
        self.initial_dialog_id = "WFDialog"

    async def welcome_step(self, step_context: WaterfallStepContext) -> DialogTurnResult:
        if not self._luis_recognizer.is_configured:
            await step_context.context.send_activity(
                MessageFactory.text(
                    "NOTE: LUIS is not configured. To enable all capabilities, add 'LuisAppId', 'LuisAPIKey' and "
                    "'LuisAPIHostName' to the appsettings.json file.",
                    input_hint=InputHints.ignoring_input,
                )
            )

            return await step_context.next(None)
        # prompt_options = PromptOptions( prompt=MessageFactory.text("In order to proceed forward please choose any
        # option to continue"), retry_prompt=MessageFactory.text("Please choose an option from the list."),
        # choices=[Choice("Task Details"), Choice("Asset Details"), Choice("Similar service records"), Choice("FAQ")]
        # ) return await step_context.prompt(ChoicePrompt.__name__, prompt_options)
        user_profile = await self.user_profile_accessor.get(step_context.context, UserProfile())
        if not user_profile.start:
            reply = MessageFactory.text("In order to proceed forward please choose any option to continue")
            user_profile.start = True
        else:
            reply = MessageFactory.text("Is there anything else I can help you with?")
        reply.suggested_actions = SuggestedActions(
            actions=[
                CardAction(
                    title="Task Details",
                    type=ActionTypes.im_back,
                    value="Task Details"
                ), CardAction(
                    title="Asset Details",
                    type=ActionTypes.im_back,
                    value="Asset Details"
                ), CardAction(
                    title="Similar service records",
                    type=ActionTypes.im_back,
                    value="Similar service records"
                ),
                CardAction(
                    title="FAQ's",
                    type=ActionTypes.im_back,
                    value="FAQ's"
                )
            ]
        )
        return await step_context.prompt(
            TextPrompt.__name__, PromptOptions(prompt=reply)
        )

    async def options_step(self, step_context: WaterfallStepContext) -> DialogTurnResult:
        if not self._luis_recognizer.is_configured:
            await step_context.context.send_activity("My Luis is not configured")
            return await step_context.end_dialog()

        intent, luis_result = await LuisHelper.execute_luis_query(self._luis_recognizer, step_context.context)

        if intent == Intent.task.value:
            return await step_context.begin_dialog(self.task_details_id, UserProfile())
        elif intent == Intent.qna.value:
            return await step_context.begin_dialog(self.faq_id, UserProfile())
        elif intent == Intent.asset.value:
            await step_context.context.send_activity("asset")
        elif intent == Intent.similar_service.value:
            await step_context.context.send_activity("similar_service")
        else:
            didnt_understand_text = (
                "Sorry, I didn't get that. Please try asking in a different way"
            )
            await step_context.context.send_activity(didnt_understand_text)

        return await step_context.next(None)

    async def final_step(self, step_context: WaterfallStepContext) -> DialogTurnResult:

        prompt_message = "What else can I do for you?"
        return await step_context.replace_dialog(self.initial_dialog_id, prompt_message)
