import os
import sys
from configparser import BasicInterpolation, ConfigParser

from dotenv import load_dotenv

# Configuration File Constants
from pydantic import BaseModel

_application_conf = "./conf/application.conf"
_default_conf = "./config.env"

load_dotenv(dotenv_path=_default_conf)


class EnvInterpolation(BasicInterpolation):
    """
    Interpolation which expands environment variables in values.
    """

    def before_get(self, parser, section, option, value, defaults):
        value = super().before_get(parser, section, option, value, defaults)

        if not os.path.expandvars(value).startswith("$"):
            return os.path.expandvars(value)
        else:
            return


try:
    config = ConfigParser(interpolation=EnvInterpolation())
    config.read(_application_conf)
except Exception as e:
    print(f"Error while loading the config: {e}")
    print("Failed to Load Configuration. Exiting!!!")
    sys.exit()


class Logging:
    level = config.get("LOGGING", "level", fallback="INFO")
    level = level if level else "INFO"
    tb_flag = config.getboolean("LOGGING", "traceback", fallback=True)
    tb_flag = tb_flag if tb_flag is not None else True


# Configuration Variables
# Kairos Configuration Variables
KAIROS_DB_HOST = config["KAIROS_DB"]["uri"]

# Postgres Configuration Variables
POSTGRES_URI = config["POSTGRES"]["postgres_uri"]
# Kafka Configuration Variables
KAFKA_HOST = config["KAFKA"]["kafka_host"]
KAFKA_PORT = config["KAFKA"]["kafka_port"]
KAFKA_TOPIC = config["KAFKA"]["kafka_topic"]

MONGO_URI = config["MONGO"]["mongo_uri"]
MLFLOW_TRACKING_URI = config["MLFLOW"]["mlflow_tracking_uri"]


class MQTTConfig(BaseModel):
    host: str = config.get("MQTT", "host")
    port: int = config.getint("MQTT", "port")
    username: str = config.get("MQTT", "username")
    password: str = config.get("MQTT", "password")
    ssl: bool = config.getboolean("MQTT", "ssl")
    connection_type: str = config.get("MQTT", "connection_type")
