import traceback
from scripts.common.config_parser import config
from scripts.common.constants import MergeWeatherDataConstants, ComponentExceptions
from scripts.common.logsetup import logger
from datetime import date, datetime
import uuid
import json
import os
import pandas as pd


class MergeWeatherData:
    def __init__(self, query):
        self.query = query
        self.historical_data = self.query['historic_data']
        self.component_input_dir = self.query['component_input_dir']
        self.component_output_dir = self.query['component_output_dir']
        self.energy_data_path = os.path.join(self.component_input_dir, os.listdir(self.component_input_dir)[0])
        # "Epoch Time"
        self.epoch_time = MergeWeatherDataConstants.EPOCH_TIME

    def preprocess_energy_data(self, energy_df):
        logger.info("Cleaning the energy data ..............")
        energy_df["Date"] = energy_df[self.epoch_time].apply(self.fix_epoch_time)
        energy_df["Date"] = pd.to_datetime(energy_df["Date"])
        energy_df.drop(columns=[self.epoch_time], inplace=True)
        return energy_df

    def merge_data_frames(self, hist_df, energy_df):
        logger.info("Merging the historical and energy data ............")
        data_train_df = pd.merge(energy_df, hist_df, left_on="Date", right_on="Date", how="inner")
        return data_train_df

    def fix_epoch_time(self, x):
        x = str(x)
        return datetime.fromtimestamp(int(x[0:10])).date()

    def preprocess_merged_data(self, data_train_df):
        try:
            logger.info("Cleaning the Merged data ..............")
            data_train_df['month'] = pd.DatetimeIndex(data_train_df['Date']).month
            data_train_df.drop(columns=["Date"], inplace=True)
            return data_train_df
        except Exception as e:
            logger.info(traceback.format_exc())
            logger.info(e)

    def split_data(self, data_train_df):
        try:
            logger.info("Creating X train and Y train ..............")
            x_train_df = data_train_df.drop(columns=["value"])
            y_train_df = data_train_df[["value"]]
            x_train_df.to_csv(os.path.join(self.component_output_dir, "X_train.csv"))
            y_train_df.to_csv(os.path.join(self.component_output_dir, "Y_train.csv"))
            return True
        except Exception as e:
            logger.info(traceback.format_exc())
            logger.info(e)

    def preprocess_historical_data(self, hist_df):
        try:
            logger.info("Cleaning the historical data ..............")
            hist_df.reset_index(inplace=True)
            hist_df["Date"] = pd.to_datetime(hist_df["Date time"])
            # print(hist_df.columns)
            hist_df.drop(
                columns=["index", "Wind Chill", "Wind Gust", "Name", "Date time", "Snow Depth", "Conditions",
                         "Heat Index", "Address", "Dew Point", "Sea Level Pressure", "Weather Type", "Longitude",
                         "Latitude", "Resolved Address", "Info","Precipitation Cover"], inplace=True)
            return hist_df
        except Exception as e:
            logger.info(traceback.format_exc())
            logger.info(e)

    def run(self):
        try:
            logger.info("Reading historical data from {}".format(self.historical_data))
            hist_df = pd.read_csv(self.historical_data)
            hist_df = self.preprocess_historical_data(hist_df)
            print(hist_df.head())
            logger.info("Reading energy data from {}".format(self.energy_data_path))
            energy_df = pd.read_csv(self.energy_data_path)[["timestamp", "value"]]
            energy_df = self.preprocess_energy_data(energy_df)
            print(energy_df.head())
            data_train_df = self.merge_data_frames(hist_df, energy_df)
            data_train_df = self.preprocess_merged_data(data_train_df)
            print(data_train_df.head())
            if self.split_data(data_train_df):
                return True
            else:
                return False
        except Exception as e:
            logger.info(traceback.format_exc())
            logger.info(e)
            return False


if __name__ == '__main__':
    try:
        obj = MergeWeatherData(config)
        if obj.run():
            logger.info("Component ran successfully")
        else:
            logger.info("Component Failed........")
    except Exception as e:
        logger.info("Merge Weather Data Component Failed")
        logger.info(traceback.format_exc())
