# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License.

from botbuilder.dialogs import (
    ComponentDialog,
    WaterfallDialog,
    WaterfallStepContext,
    DialogTurnResult,
)
from botbuilder.dialogs.prompts import TextPrompt, PromptOptions
from botbuilder.core import MessageFactory, TurnContext
from botbuilder.schema import InputHints

from booking_details import BookingDetails
from flight_booking_recognizer import FlightBookingRecognizer
from helpers.luis_helper import LuisHelper, Intent
from .safetychecklist import safetychecklist
from .faq import faq
from botbuilder.schema import ChannelAccount, CardAction, ActionTypes, SuggestedActions
from botbuilder.dialogs.prompts import ConfirmPrompt, ChoicePrompt, PromptOptions
from botbuilder.dialogs.choices import Choice

class MainDialog(ComponentDialog):
    def __init__(
            self, luis_recognizer: FlightBookingRecognizer, safetychecklist: safetychecklist, faq: faq
    ):
        super(MainDialog, self).__init__(MainDialog.__name__)

        self._luis_recognizer = luis_recognizer
        self.safetychecklist_id = safetychecklist.id
        self.faq_id = faq.id
        self.add_dialog(TextPrompt(TextPrompt.__name__))
        self.add_dialog(safetychecklist)
        self.add_dialog(faq)
        self.add_dialog(
            WaterfallDialog(
                "WFDialog", [self.intro_step, self.act_step, self.final_step]
            )
        )

        self.initial_dialog_id = "WFDialog"

    async def intro_step(self, step_context: WaterfallStepContext) -> DialogTurnResult:
        if not self._luis_recognizer.is_configured:
            await step_context.context.send_activity(
                MessageFactory.text(
                    "NOTE: LUIS is not configured. To enable all capabilities, add 'LuisAppId', 'LuisAPIKey' and "
                    "'LuisAPIHostName' to the appsettings.json file.",
                    input_hint=InputHints.ignoring_input,
                )
            )

            return await step_context.next(None)

        reply = MessageFactory.text("Hi Welcome to iLens Assistant!! please select any option to proceed forward")

        reply.suggested_actions = SuggestedActions(
            actions=[
                CardAction(
                    title="Task Details",
                    type=ActionTypes.im_back,
                    value="Task Details"
                ),
                CardAction(
                    title="FAQ's",
                    type=ActionTypes.im_back,
                    value="FAQ's"
                )
            ]
        )
        return await step_context.prompt(
            TextPrompt.__name__, PromptOptions(prompt=reply)
        )

    async def act_step(self, step_context: WaterfallStepContext) -> DialogTurnResult:
        if not self._luis_recognizer.is_configured:
            await step_context.context.send_activity("luis is not there")
            # LUIS is not configured, we just run the BookingDialog path with an empty BookingDetailsInstance.
            return await step_context.begin_dialog(
                self._booking_dialog_id, BookingDetails()
            )

        # Call LUIS and gather any potential booking details. (Note the TurnContext has the response to the prompt.)
        intent, luis_result = await LuisHelper.execute_luis_query(
            self._luis_recognizer, step_context.context
        )
        print(intent)
        if intent == Intent.task.value:
            return await step_context.begin_dialog(self.safetychecklist_id, luis_result)
        elif intent == Intent.qna.value:
            return await step_context.begin_dialog(self.faq_id, luis_result)
        else:
            didnt_understand_text = (
                "Sorry, I didn't get that. Please try asking in a different way"
            )
            didnt_understand_message = MessageFactory.text(
                didnt_understand_text, didnt_understand_text, InputHints.ignoring_input
            )
            await step_context.context.send_activity(didnt_understand_message)

        return await step_context.next(None)

    async def final_step(self, step_context: WaterfallStepContext) -> DialogTurnResult:
        # If the child dialog ("BookingDialog") was cancelled or the user failed to confirm,
        # the Result here will be null.
        prompt_message = "What else can I do for you?"
        return await step_context.replace_dialog(self.id, prompt_message)
