import yaml
import kfp
import logging
import os
import json
from dotenv import load_dotenv

load_dotenv('pipeline-conf.env')
logging.basicConfig(level=logging.INFO, format='%(asctime)s :: %(levelname)s :: %(message)s')

deployment_yaml = os.getenv("DEPLOYMENT_YAML")
kubeflow_uri = os.getenv("KUBEFLOW_URI")
login_token = os.getenv("LOGIN_TOKEN")
pipeline_version = os.getenv("PIPELINE_VERSION")
experiment_name = os.getenv("EXPERIMENT_NAME")
cron_expression = os.getenv("CRON_EXPRESSION")
conf_path = os.getenv("CONF_LIST", '')
variables = os.getenv('VARIABLES', 'false')
is_recurring_run = os.environ.get("RECURRING_RUN", 'false')

client = kfp.Client(host=kubeflow_uri, cookies=f'login-token={login_token}', namespace='project-099')


def pipeline_config():
    """
    Function used to configure the Pipeline
    :return: Uploads and Triggers the run in the kubeflow
    """
    try:
        run_name = os.getenv('RUN_NAME', '')
        pipeline_name = os.getenv('PIPELINE_NAME', '')
        if conf_path:
            with open(conf_path, 'r') as f:
                # Load the JSON data from the file
                data = json.load(f)
                config_list = data.get("plant_details")
                logging.info("Loading the Plant JSON Data")

        else:
            config_list = [{'pipeline_name': pipeline_name, 'run_name': run_name}]

        for each_config in config_list:
            pipeline_name = each_config.get('pipeline_name', 'default_pipeline')
            run_name = each_config.get('run_name', 'default_run')
            pipeline_version_name = f"{pipeline_name}-{pipeline_version}"
            pipeline_id = client.get_pipeline_id(pipeline_name)

            # Pipeline name
            pipeline_id = upload_pipeline(
                pipeline_id=pipeline_id, pipeline_name=pipeline_name, pipeline_version_name=pipeline_version_name)

            experiment = client.create_experiment(experiment_name)
            logging.info(f"Creating Experiment with the name {experiment_name}")
            get_recurring_runs = client.list_recurring_runs(experiment_id=experiment.id, page_size=100)
            runs_list = list()
            if get_recurring_runs.jobs:
                runs_list = get_recurring_runs.jobs
            for each_run in runs_list:
                if each_run.name == run_name:
                    client.disable_job(each_run.id)

            # Open the YAML file and load its contents into a Python dictionary
            if variables == "true" and os.path.exists('variables.yml'):
                final_json = add_pipeline_param()
            else:
                final_json = dict()

            plant_info = each_config.get('plant_info', {})
            logging.info(f"Getting plant_info {plant_info}")
            if plant_info:
                final_json["plant_info"] = plant_info
                logging.info("plant_info is existing")

            trigger_run(
                experiment_id=experiment.id, run_name=run_name, final_json=final_json, pipeline_id=pipeline_id)
    except Exception as e:
        logging.exception(f"Unable to Trigger the Kubeflow env {e}")


def resource_allocation():
    """
    This Function is used to allocate the resources for each component in pipeline
    :return:
    """
    try:
        memory_limit = os.getenv("MEMORY_LIMIT", '0M')
        cpu_limit = os.getenv("CPU_LIMIT", '0m')
        memory_request = os.getenv("MEMORY_REQUEST", '0M')
        cpu_request = os.getenv('CPU_REQUEST', '0m')
        with open(deployment_yaml, 'r') as f:
            data = yaml.safe_load(f)
        components_list = data.get('spec', {}).get('templates', [])
        for each_comp in components_list:
            if 'container' in each_comp:
                each_comp['container']['resources'] = {"limits": {"memory": memory_limit, "cpu": cpu_limit},
                                                       "requests": {"memory": memory_request, "cpu": cpu_request}}
        data['spec']['templates'] = components_list
        with open(deployment_yaml, 'w') as file:
            yaml.dump(data, file)
    except Exception as e:
        logging.exception(f"Unable to allocate resources {e}")


def add_pipeline_param():
    """
    This Function take the variables.yml and pass them as pipeline param for pipeline execution
    :return:
    """
    try:
        logging.info("variables.yml file is existing")
        with open('variables.yml', 'r') as yaml_file:
            yaml_data = yaml.load(yaml_file, Loader=yaml.FullLoader)
        pipeline_param = dict()
        for each in yaml_data.get("deployment", []).get("environmentVar", []):
            if 'valueFrom' not in list(each.keys()):
                pipeline_param[each.get('name')] = each.get('value')
        final_json = dict()
        final_json["pipeline_param"] = pipeline_param
        logging.info("Adding pipeline parameters")
        return final_json
    except Exception as e:
        logging.exception(f"Unable to add the pipeline param {e}")


def upload_pipeline(pipeline_id, pipeline_version_name, pipeline_name):
    """
    Function is to used to upload the pipeline in kubeflow
    :param pipeline_id: Pipeline id
    :param pipeline_version_name: Pipeline version name
    :param pipeline_name: Pipeline Name
    :return: Returns Uploaded pipeline ID
    """
    try:
        if pipeline_id:
            pipeline_list = client.list_pipeline_versions(pipeline_id=pipeline_id, page_size=100)
            if pipeline_list.versions:
                for each_version in pipeline_list.versions:
                    if each_version.name == pipeline_version_name:
                        client.delete_pipeline_version(each_version.id)
            client.upload_pipeline_version(
                pipeline_package_path=deployment_yaml, pipeline_version_name=pipeline_version_name,
                pipeline_name=pipeline_name)
            logging.info(
                f"Uploaded Pipeline version with pipeline name {pipeline_name} and pipeline version {pipeline_version_name}")

        else:
            pipeline = client.upload_pipeline(deployment_yaml, pipeline_name=pipeline_name)
            pipeline_id = pipeline.id
            logging.info(f"Uploaded Pipeline version with pipeline name {pipeline_name}")
        return pipeline_id
    except Exception as e:
        logging.exception(f"Unable to upload the pipeline {e}")


def trigger_run(experiment_id, run_name, final_json, pipeline_id):
    """
    Function is used to trigger the Run
    :param experiment_id: Experiment Id
    :param run_name: Run name
    :param final_json: json where the pipeline params are present
    :param pipeline_id: Pipeline Id
    :return: Creates and triggers the Run
    """
    try:
        if is_recurring_run == "true":
            logging.info("Recurring run")
            logging.info("Starting to create an recurring run")
            client.create_recurring_run(
                experiment_id=experiment_id, job_name=run_name, pipeline_id=pipeline_id,
                cron_expression=cron_expression, params=final_json)
            logging.info(f"Successfully Triggered the Recurring Run with run name as {run_name}")

        else:
            client.run_pipeline(experiment_id, run_name, pipeline_id=pipeline_id, params=final_json)
            logging.info(f"Successfully Triggered Run with run name as {run_name}")
    except Exception as e:
        logging.info(f"Unable to trigger the Run {e}")


if __name__ == "__main__":
    is_allocation = os.getenv('IS_ALLOCATION', 'false')
    if is_allocation == "true":
        resource_allocation()
    pipeline_config()
