import traceback
from scripts.common.config_parser import *
from scripts.common.constants import AddtoModelStoreConstants, ComponentExceptions
from scripts.common.logsetup import logger
from sklearn.ensemble import RandomForestRegressor
import pickle
from azure.storage.blob import BlobServiceClient
import os.path
from os import path
from datetime import date
import uuid


class AddtoModelStore:
    def __init__(self, query):
        self.query = query
        self.container = self.query["container_name"]
        self.connection_string = self.query["connection_string"]
        self.blob_service_client = BlobServiceClient.from_connection_string(self.connection_string)
        self.container_client = self.blob_service_client.get_container_client(self.container)
        self.ilens_tag_hierarchy = self.query['ilens_tag_hierarchy']
        self.component_input_dir = self.query['component_input_dir']
        self.artifact_current_path = self.query['artifact_current_path']
        self.artifact_archived_path = self.query['artifact_archived_path']
        self.new_model_path = os.path.join(self.component_input_dir, os.listdir(self.component_input_dir)[0])
        self.new_model_name = os.listdir(self.component_input_dir)[0]
        self.meta_data_file = self.query['metadata_file']
        self.update_current_model = {
            "model_name": "randomforest",
            "model_params": None,
            "training_date": str(date.today()),
            "framework": None,
            "serializedObjectType": "pkl",
            "model_fl_name": self.new_model_name
        }
        self.first_run = [
            {
                'id': self.ilens_tag_hierarchy,
                'current_model': {
                    'model_name': 'randomforest',
                    'model_params': None,
                    'training_date': str(date.today()),
                    'framework': None,
                    'serializedObjectType': 'pkl',
                    'model_fl_name': self.new_model_name
                },
                'archived': [

                ]
            }
        ]

    def move_files(self, file_name):
        try:
            logger.info("started moving files from current to archived folder......")
            source_blob_path = os.path.join(self.artifact_current_path, file_name)
            dest_blob_path = os.path.join(self.artifact_archived_path, file_name)
            source_blob = self.blob_service_client.get_blob_client(self.container, source_blob_path)
            dest_blob = self.blob_service_client.get_blob_client(self.container, dest_blob_path)
            dest_blob.start_copy_from_url(source_blob.url, requires_sync=True)
            copy_properties = dest_blob.get_blob_properties().copy
            logger.info("File at " + source_blob_path + " is moved to " + dest_blob_path)
            if copy_properties.status != "success":
                dest_blob.abort_copy(copy_properties.id)
                raise Exception(
                    f"Unable to copy blob %s with status %s"
                    % (source_blob_path, copy_properties.status)
                )
            source_blob.delete_blob()
        except Exception as e:
            logger.info(traceback.format_exc())
            raise Exception(e)

    def check_meta_data(self):
        return path.exists(self.query['metadata_file'])

    def upload_to_blob(self):
        try:
            logger.info("started uploading data..............")
            blob_client = self.blob_service_client.get_blob_client(container=self.container,
                                                                   blob=os.path.join(self.artifact_current_path,
                                                                                     self.new_model_name))
            with open(self.new_model_path, "rb") as data:
                blob_client.upload_blob(data, overwrite=True)
            logger.info("File at " + self.new_model_path + " is uploaded to " + os.path.join(self.artifact_current_path,
                                                                                             self.new_model_name))
            return True
        except Exception as e:
            logger.info(traceback.format_exc())
            raise Exception(e)

    def update_meta_data(self, data):
        try:
            logger.info("Updating metadata json file...............")
            for i in data:
                if i['id'] == self.ilens_tag_hierarchy:
                    i['archived'].append(i['current_model'])
                    i['current_model'] = self.update_current_model
                else:
                    pass
            # print(data)
            return data
        except Exception as e:
            logger.info(traceback.format_exc())
            raise Exception(e)

    def get_file_name_from_meta_data(self, data):
        try:
            logger.info("Reading metadata json file to get the old models name....")
            for i in data:
                if i['id'] == self.ilens_tag_hierarchy:
                    return i['current_model']['model_fl_name']
        except Exception as e:
            logger.info(traceback.format_exc())
            raise Exception(e)

    def run(self):
        try:
            checK_meta_file = self.check_meta_data()
            # print(checK_meta_file)
            if checK_meta_file:
                logger.info("metadata json file is present.........")
                try:
                    logger.info("Reading Json file")
                    with open(self.meta_data_file) as f:
                        data = json.load(f)
                except:
                    logger.info("Failed reading Json File")
                    logger.info(traceback.format_exc())
                old_model_name = self.get_file_name_from_meta_data(data)
                self.move_files(old_model_name)
                self.upload_to_blob()
                data = self.update_meta_data(data)
                json_object = json.dumps(data, indent=4)
                with open(self.query['metadata_file'], "w") as outfile:
                    outfile.write(json_object)
            else:
                logger.info(
                    "metadata json file is not present so created metadata json file at " + self.query['metadata_file'])
                if self.upload_to_blob():
                    json_object = json.dumps(self.first_run, indent=4)
                    with open(self.query['metadata_file'], "w") as outfile:
                        outfile.write(json_object)
                else:
                    logger.info("AddToModelStore component failed...............")
            return True
        except Exception as e:
            logger.info(traceback.format_exc())
            raise Exception(e)


if __name__ == '__main__':
    try:
        obj = AddtoModelStore(config)
        obj.run()
    except:
        logger.info("Model Object Component Failed")
        logger.info(traceback.format_exc())
