from __future__ import annotations

import base64
from datetime import datetime, timedelta

import jwt
from Cryptodome.Cipher import AES
from passlib.context import CryptContext
from validate_email import validate_email

from scripts.config import Services
from scripts.database.mongo.mongo_login import MongoUser
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger


class NormalLogin:
    def __init__(self):
        self.mongo_user = MongoUser()
        self.pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")
        self.db_user_data = None
        self.db_user_data = None
        self.dt = datetime.now()
        self.time_dt = datetime.now()

    @staticmethod
    def un_pad(s):
        # un_padding the encrypted password
        return s[:-ord(s[len(s) - 1:])]

    def password_decrypt(self, password):
        try:
            # encoding the Key
            key = Services.KEY_ENCRYPTION.encode(Services.ENCODING_TYPE)
            # decoding the received password
            enc = base64.b64decode(password)
            # mode for the decryption
            mode = AES.MODE_CBC
            # getting the initialization vector
            iv = enc[:AES.block_size]
            # decoding with AES
            cipher = AES.new(key, mode, iv)
            # decoding the password
            data = cipher.decrypt(enc[AES.block_size:])
            if len(data) == 0:
                raise ValueError("Decrypted data is empty")
            # removing the padding to get the password
            data = self.un_pad(data)
            return data.decode('utf-8')
        except Exception as e:
            logger.exception(e)

    @staticmethod
    def user_data_validation(username, password) -> dict | None:
        try:
            # checking for valid username
            if username == "" or username == "user@example.com" or validate_email(username) is not True:
                return {"message": ErrorMessages.ERROR_INVALID_USERNAME, "data": username}
            # checking for valid password
            if password == "" or password == "string":
                return {"message": ErrorMessages.ERROR_INVALID_PASSWORD, "data": password}
            return None
        except Exception as e:
            logger.exception(e)

    def db_data_validation(self, project_id, login_data):
        try:
            # fetching the data based on the username
            self.db_user_data = MongoUser().fetch_user_details(login_data["username"])
            # if the user is not available
            if not self.db_user_data:
                return False, {"message": ErrorMessages.ERROR_UNAUTHORIZED_USER_LOGIN,
                               "data": {"username": login_data["username"]}}
            # Check the project id from the request body
            if self.db_user_data["project_id"] != Services.PROJECT_ID or Services.PROJECT_ID != project_id:
                return False, {"message": ErrorMessages.ERROR_UNAUTHORIZED_USER_LOGIN, "data": login_data.username}
            # if the user exist
            return None, {"message": True}
        except Exception as e:
            logger.exception(e)

    def db_password_matching(self, project_id, login_data, password):
        try:
            # getting the response after checking for the user data in db
            response, message = self.db_data_validation(project_id, login_data)
            # if the response is false then an error message is send back
            if response is not None:
                return response, message
            # if the user exists in db then password is matched
            if not self.pwd_context.verify(password, self.db_user_data["password"]):
                return False, {"message": ErrorMessages.ERROR_PASSWORD_MISMATCH,
                               "data": {"username": login_data["username"]}}
            # if the password is correct
            return None, {"username": login_data["username"], "role": self.db_user_data["user_role"]}
        except Exception as e:
            logger.exception(e)

    @staticmethod
    def create_access_token(data: dict, expires_delta):
        try:
            # creating a copy of the data
            to_encode = data.copy()
            # checking if the expires_delta is empty
            if expires_delta:
                expire = datetime.utcnow() + expires_delta
            else:
                expire = datetime.utcnow() + timedelta(minutes=15)
            # updating the data with the expiration time
            to_encode.update({"expire": expire.isoformat()})
            # creating the token
            encoded_jwt = jwt.encode(to_encode, Services.SECRET_KEY, algorithm=Services.ALGORITHM)
            return encoded_jwt
        except Exception as e:
            logger.exception(e)

    def generate_tokens(self, login_data, data):
        try:
            # creating the expiration time
            access_token_expires = timedelta(minutes=Services.ACCESS_TOKEN_EXPIRE_MINUTES)
            # creating the access token
            access_token = self.create_access_token(
                data={"username": login_data["username"], "role": data["role"]}, expires_delta=access_token_expires
            )
            if access_token:
                return {"access_token": access_token, "token_type": "bearer"}
            else:
                return None
        except Exception as e:
            logger.exception(e)
