from fastapi import APIRouter, HTTPException, status, Depends, Request
from fastapi.responses import JSONResponse

from scripts.constants.api import ApiEndPoints
from scripts.core.handlers.dashboard_handler import DashboardManagement
from scripts.core.handlers.login_handler import LoginHandlers
from scripts.core.handlers.user_management_handler import UserManagement
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger
from scripts.schemas.default_responses import DefaultFailureResponse
from scripts.schemas.project_schema import LoginRequest, UserActions, EmailValidation, UserIDValidation, ResetPassword
from scripts.utils.security.authorize_access import AuthorizeAccess
from scripts.utils.security.decorators import MetaInfoSchema, auth

# creating the login api
router = APIRouter()
# initializing the handler
obj_login_handler = LoginHandlers()
obj_user_handler = UserManagement()
obj_dashboard_handler = DashboardManagement()


# login API
@router.post(ApiEndPoints.asset_manager_submit)
async def login_default_api(
        user_data: LoginRequest,
        request: Request,
):
    try:
        # mapper for login types
        login_mapper = {
            "general_login": obj_login_handler.general_login,
            "google": obj_login_handler.google_login,
            "microsoft": obj_login_handler.microsoft_login
        }

        # getting the functions based on the login types
        if user_data.login_type in login_mapper:
            return login_mapper[user_data.login_type](user_data, request)
        else:
            return HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=ErrorMessages.ERROR_INVALID_REQUEST)

    except Exception as e:
        logger.exception(e)
        return JSONResponse(
            content=DefaultFailureResponse(status="failed",
                                           message=ErrorMessages.OP_FAILED).dict(),
            status_code=status.HTTP_200_OK)


# Forgot Password API
@router.post(ApiEndPoints.asset_manager_forgot)
async def forgot_password(
        validation_data: EmailValidation
):
    try:
        # forgot password
        response = obj_login_handler.forgot_password_handler(validation_data.email)
        return response
    except Exception as e:
        logger.exception(e)
        return JSONResponse(
            content=DefaultFailureResponse(status="failed",
                                           message=ErrorMessages.OP_FAILED).dict(),
            status_code=status.HTTP_200_OK)


# Reset Password API for sending email
@router.get(ApiEndPoints.asset_manager_reset)
async def reset_password(
        request: Request
):
    try:
        # Get the JWT token from the query parameters
        response = obj_login_handler.validate_jwt(request)
        return response
    except Exception as e:
        logger.exception(e)
        return JSONResponse(
            content=DefaultFailureResponse(status="failed",
                                           message=ErrorMessages.OP_FAILED).dict(),
            status_code=status.HTTP_200_OK)


# Reset Password API
@router.post(ApiEndPoints.asset_manager_reset)
async def reset_password(
        reset_data: ResetPassword
):
    try:
        # Get the JWT token from the query parameters
        response = obj_login_handler.reset_user_password(reset_data)
        return response
    except Exception as e:
        logger.exception(e)
        return JSONResponse(
            content=DefaultFailureResponse(status="failed",
                                           message=ErrorMessages.OP_FAILED).dict(),
            status_code=status.HTTP_200_OK)


# User Management API
@router.post(ApiEndPoints.asset_manager_user_table_actions)
async def user_management(
        user_data: UserActions,
        request: MetaInfoSchema = Depends(auth)
):
    try:
        response = AuthorizeAccess().admin_authorize(request)
        if not response:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_UNAUTHORIZED_ACCESS).dict(),
                status_code=status.HTTP_200_OK)
        # mapper for registration types
        register_mapper = {
            "general_login": obj_user_handler.general_register,
            "google": obj_user_handler.google_register,
            "microsoft": obj_user_handler.microsoft_register
        }

        if user_data.action == "addnew" and user_data.login_type in register_mapper:
            # registration
            return register_mapper[user_data.login_type](user_data)
        elif user_data.action == "edit":
            # updating
            return obj_user_handler.update_user_details(user_data)
        elif user_data.action == "delete":
            # deleting
            return obj_user_handler.delete_user_details(user_data.user_id)
        else:
            return HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=ErrorMessages.ERROR_INVALID_REQUEST)
    except Exception as e:
        logger.exception(e)
        return JSONResponse(
            content=DefaultFailureResponse(status="failed",
                                           message=ErrorMessages.OP_FAILED).dict(),
            status_code=status.HTTP_200_OK)


# View users API Header
@router.post(ApiEndPoints.asset_manager_user_header)
async def user_view_header(
        request: MetaInfoSchema = Depends(auth)
):
    try:
        # authorize the user
        response = AuthorizeAccess().admin_authorize(request)
        if not response:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_UNAUTHORIZED_ACCESS).dict(),
                status_code=status.HTTP_200_OK)
        # getting the header for the user table
        response = obj_user_handler.fetch_view_header()
        if not response:
            return HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=ErrorMessages.ERROR_IN_FETCHING)
        return response
    except Exception as e:
        logger.exception(e)
        return JSONResponse(
            content=DefaultFailureResponse(status="failed",
                                           message=ErrorMessages.OP_FAILED).dict(),
            status_code=status.HTTP_200_OK)


# View users API
@router.post(ApiEndPoints.asset_manager_user_view)
async def user_view_data(
        request: MetaInfoSchema = Depends(auth)
):
    try:
        # authorize the user
        response = AuthorizeAccess().admin_authorize(request)
        if not response:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_UNAUTHORIZED_ACCESS).dict(),
                status_code=status.HTTP_200_OK)
        response = obj_user_handler.fetch_user_details()
        if not response:
            return HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=ErrorMessages.ERROR_IN_FETCHING)
        return response
    except Exception as e:
        logger.exception(e)
        return JSONResponse(
            content=DefaultFailureResponse(status="failed",
                                           message=ErrorMessages.OP_FAILED).dict(),
            status_code=status.HTTP_200_OK)


# API for change password
@router.post(ApiEndPoints.asset_manager_user_reset)
async def user_change_password(
        reset_data: ResetPassword,
        request: MetaInfoSchema = Depends(auth)
):
    try:
        # authorize the user
        response = AuthorizeAccess().login_authorize(request, reset_data)
        if not response:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_UNAUTHORIZED_ACCESS).dict(),
                status_code=status.HTTP_200_OK)
        response = obj_user_handler.reset_password(reset_data)
        if not response:
            return HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=ErrorMessages.ERROR_IN_FETCHING)
        return response
    except Exception as e:
        logger.exception(f"From user change password - {e}")
        return JSONResponse(
            content=DefaultFailureResponse(status="failed",
                                           message=ErrorMessages.OP_FAILED).dict(),
            status_code=status.HTTP_200_OK)


# download Button Dashboard
@router.post(ApiEndPoints.asset_manager_dashboard_download)
async def dashboard_download_files(
        request: MetaInfoSchema = Depends(auth)
):
    try:
        response = AuthorizeAccess().login_authorize(request, None)
        if not response:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_UNAUTHORIZED_ACCESS).dict(),
                status_code=status.HTTP_200_OK)
        # getting the data for the download dashboard
        response = obj_dashboard_handler.download_header()
        if not response:
            return HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=ErrorMessages.ERROR_IN_FETCHING)
        return response
    except Exception as e:
        logger.exception(e)
        return JSONResponse(
            content=DefaultFailureResponse(status="failed",
                                           message=ErrorMessages.OP_FAILED).dict(),
            status_code=status.HTTP_200_OK)


# API for logout
@router.post(ApiEndPoints.asset_manager_user_logout)
async def user_logout(
        user_id: UserIDValidation,
        request: MetaInfoSchema = Depends(auth)
):
    try:
        # authorizing the user
        # logging out
        response = obj_user_handler.logout_user(user_id, request)
        if not response:
            return HTTPException(
                status_code=status.HTTP_403_FORBIDDEN,
                detail=ErrorMessages.ERROR_IN_FETCHING)
        return response
    except Exception as e:
        logger.exception(e)
        return JSONResponse(
            content=DefaultFailureResponse(status="failed",
                                           message=ErrorMessages.OP_FAILED).dict(),
            status_code=status.HTTP_200_OK)
