from fastapi import APIRouter, status
from fastapi.responses import JSONResponse

from scripts.constants.api import ApiEndPoints
from scripts.core.handlers.login_handler import LoginHandlers
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger
from scripts.schemas.default_responses import DefaultResponse, DefaultFailureResponse
from scripts.schemas.login_schema import NormalLogin

# creating the login api
router = APIRouter(prefix=ApiEndPoints.version)
# initializing the handler
obj_login_handler = LoginHandlers()


@router.post(ApiEndPoints.asset_manager_submit)
async def login_default(login_data: NormalLogin):
    try:
        # decrypting the password from the UI
        decrypted_password = obj_login_handler.password_decrypt(login_data.password)
        # validating the received inputs empty or not
        response = obj_login_handler.user_data_validation(login_data.username, decrypted_password)
        # Account is not registered
        if response is not None:
            return JSONResponse(content=DefaultFailureResponse(error=response["message"]).dict(),
                                status_code=status.HTTP_400_BAD_REQUEST)
        # checking for the account and password matching
        response, data = obj_login_handler.db_password_matching(login_data, decrypted_password)
        # if the user is not valid
        if response is not None and data["message"] == ErrorMessages.ERROR_UNAUTHORIZED_USER_LOGIN:
            return JSONResponse(content=DefaultFailureResponse(error=data).dict(),
                                status_code=status.HTTP_404_NOT_FOUND)
        # if the passwords doesn't match with the db data
        if response is not None:
            return JSONResponse(content=DefaultFailureResponse(error=data).dict(),
                                status_code=status.HTTP_401_UNAUTHORIZED)
        #  generating the access tokens
        response = obj_login_handler.generate_tokens(login_data, data)
        # token generation unsuccessful
        if response is None:
            return JSONResponse(
                content=DefaultFailureResponse(message="Access Unsuccessful",
                                               error=ErrorMessages.ERROR_TOKEN_GENERATION).dict(),
                status_code=status.HTTP_403_FORBIDDEN)
        # sending successful response to UI
        return JSONResponse(
            content=DefaultResponse(message="Login Successful", payload=response).dict(),
            status_code=status.HTTP_200_OK)
    except Exception as e:
        logger.exception(e)
