from __future__ import annotations

import base64

from Cryptodome.Cipher import AES
from passlib.context import CryptContext

from scripts.config import Services
from scripts.database.mongo.mongo_login import MongoUser
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger


class LoginHandlers:
    def __init__(self):
        self.pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")
        self.db_user_data = None

    @staticmethod
    def un_pad(s):
        return s[:-ord(s[len(s) - 1:])]

    def password_decrypt(self, password):
        try:
            # encoding the Key
            key = Services.KEY_ENCRYPTION.encode('utf-8')
            # decoding the received password
            enc = base64.b64decode(password)
            # getting the initialization vector
            iv = enc[:AES.block_size]
            # decoding with AES
            cipher = AES.new(key, AES.MODE_CBC, iv)
            # decoding the password
            data = cipher.decrypt(enc[AES.block_size:])
            if len(data) == 0:
                raise ValueError("Decrypted data is empty")
            # removing the padding to get the password
            data = self.un_pad(data)
            return data.decode('utf-8')
        except Exception as e:
            logger.exception(e)

    @staticmethod
    def user_data_validation(username, password) -> dict | None:
        try:
            # checking for valid username
            if username == "" or username == "user@example.com":
                return {"message": ErrorMessages.ERROR_INVALID_USERNAME, "data": username}
            # checking for valid password
            if password == "" or password == "string":
                return {"message": ErrorMessages.ERROR_INVALID_PASSWORD, "data": password}
            return None
        except Exception as e:
            logger.exception(e)

    def db_data_validation(self, username):
        try:
            # fetching the data based on the username
            self.db_user_data = MongoUser().fetch_user_details(username)
            # if the user is not available
            if not self.db_user_data:
                return False, {"message": ErrorMessages.ERROR_USER_NOT_REGISTERED, "data": username}
            # if the user exist
            return None, {"message": True}
        except Exception as e:
            logger.exception(e)

    def db_password_matching(self, username, password):
        try:
            # getting the response after checking for the user data in db
            response, message = self.db_data_validation(username)
            # if the response is false then a error message is send back
            if response is not None:
                return response, message
            # if the user exists in db then password is matched
            if not self.pwd_context.verify(password, self.db_user_data["password"]):
                return False, {"message": ErrorMessages.ERROR_PASSWORD_MISMATCH, "data": {"username": username}}
            # if the password is correct
            return None, {"username": username, "role": self.db_user_data["user_role"]}
        except Exception as e:
            logger.exception(e)
