import datetime

from scripts.database.mongo.mongo_db import MongoUser
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger
from fastapi.responses import JSONResponse
from fastapi import status
from scripts.schemas.default_responses import DefaultResponse, DefaultFailureResponse
from scripts.utils.security.password_util import EncryptDecryptPassword
from scripts.utils.validations_util import UserDataValidations

obj_mongo_user = MongoUser()


class UserManagement:
    def __init__(self):
        self.method = "register"

    # for normal registration using email and password
    def normal_register(self, user_data):
        try:
            response, message = UserDataValidations.register_data_validation(user_data, 'normal', self.method)
            if not response:
                return message
            # fetching the data based on the username
            db_user_data = obj_mongo_user.fetch_one_user_details(user_data.email)
            # if the user is not available
            if db_user_data:
                return {"message": ErrorMessages.ERROR_EMAIL_EXIST,
                        "data": {"username": user_data.email}}
            created_at = datetime.datetime.now()
            updated_at = datetime.datetime.now()
            reg_time = {"created_at": created_at, "updated_at": updated_at}
            user_data_dict = {key: (EncryptDecryptPassword().password_encrypt(value)
                                    if key == "password" else value) for key, value in user_data}
            user_data_reg = user_data_dict | reg_time
            if not obj_mongo_user.insert_new_user(user_data_reg):
                return {"message": ErrorMessages.ERROR_STORING_DATA,
                        "data": {"username": user_data_dict}}
            return {"message": "New user registered",
                    "data": {"username": user_data_dict["email"]}}
        except Exception as e:
            logger.exception(e)

    @staticmethod
    # for Google registration using gmail
    def google_register():
        try:
            return {"message": "Not available now"}
        except Exception as e:
            logger.exception(e)

    @staticmethod
    # for microsoft registration using microsoft account
    def microsoft_register():
        try:
            return {"message": "Not available"}
        except Exception as e:
            logger.exception(e)

    # update user details
    def update_user_details(self, email, update_data):
        try:
            self.method = "update"
            db_user_data = obj_mongo_user.fetch_one_user_details(email)
            # if the user is not available
            if db_user_data is None:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_EMAIL_ID_DOESNT_EXIST).dict(),
                    status_code=status.HTTP_404_NOT_FOUND)
            if update_data.email is not None:
                db_user_data = obj_mongo_user.fetch_one_user_details(update_data.email)
                # if the user is not available
                if db_user_data is not None:
                    return JSONResponse(
                        content=DefaultFailureResponse(status="failed",
                                                       message=ErrorMessages.ERROR_EMAIL_EXIST).dict(),
                        status_code=status.HTTP_404_NOT_FOUND)
            filter_data_updated = {"email": email}
            update_data_removed = {key: value for key, value in update_data if value is not None}
            response, message = UserDataValidations.update_data_validation(update_data)
            if not response:
                return message
            response = obj_mongo_user.update_user(filter_data_updated, update_data_removed)
            if not response:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_IN_UPDATING).dict(),
                    status_code=status.HTTP_200_OK)
            return JSONResponse(
                content=DefaultResponse(status="success", message="Updated Successfully",
                                        data=update_data_removed).dict(),
                status_code=status.HTTP_200_OK)
        except Exception as e:
            logger.exception(e)

    # delete user
    @staticmethod
    def delete_user_details(email):
        db_user_data = obj_mongo_user.fetch_one_user_details(email)
        # if the user is not available
        if db_user_data is None:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_EMAIL_ID_DOESNT_EXIST).dict(),
                status_code=status.HTTP_404_NOT_FOUND)
        filter_data_updated = {"email": email}
        response = obj_mongo_user.delete_user(filter_data_updated)
        if not response:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_IN_UPDATING).dict(),
                status_code=status.HTTP_404_NOT_FOUND)
        return JSONResponse(
            content=DefaultResponse(status="success", message="Deleted Successfully",
                                    data=filter_data_updated).dict(),
            status_code=status.HTTP_200_OK)

    @staticmethod
    def fetch_user_details():
        try:
            filter_data = {'_id': 0,
                           "login_type": 0,
                           "is_alive": 0,
                           "password": 0,
                           "created_at": 0,
                           "updated_at": 0}
            cursor_data = obj_mongo_user.fetch_all_user_details({}, filter_data)
            cursor_data_count = cursor_data.explain()
            if cursor_data_count["executionStats"]["nReturned"] <= 0:
                return None
            list_user_data = []
            for users in cursor_data:
                list_user_data.append(users)
            return JSONResponse(
                content=DefaultResponse(status="success", message="Fetched Successfully",
                                        data=list_user_data).dict(),
                status_code=status.HTTP_200_OK)
        except Exception as e:
            logger.exception(e)
