import smtplib
import ssl
from datetime import timedelta, datetime
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText

from jwt.exceptions import ExpiredSignatureError

from google.oauth2 import id_token
from google.auth.transport import requests
from google.auth.exceptions import InvalidValue

from fastapi import status
from fastapi.responses import JSONResponse, RedirectResponse

from scripts.config import Services, Secrets
from scripts.core.handlers.process_login_data import NormalLogin
from scripts.database.mongo.mongo_db import MongoUser
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger
from scripts.schemas.default_responses import DefaultFailureResponse, DefaultResponse, DefaultSuccessResponse
from scripts.utils.security.jwt_util import JWT
from scripts.utils.security.password_util import EncryptDecryptPassword

obj_mongo_user = MongoUser()
jwt = JWT()


class LoginHandlers:
    def __init__(self):
        self.obj_login_handler = NormalLogin()
        self.pass_decrypt = EncryptDecryptPassword()
        self.login_type = ""

    def normal_login(self, user_data, request):
        self.login_type = "general_login"
        # decrypting the password from the UI
        decrypted_password = self.pass_decrypt.password_decrypt(user_data.password)
        # validating the received inputs empty or not
        # password decrypted form - token "password"
        responses = self.obj_login_handler.user_data_validation(
            user_data.email,
            decrypted_password.split("\"")[1])
        # Account is not registered
        if responses is not None:
            return JSONResponse(content=DefaultFailureResponse(status="failed",
                                                               message=responses).dict(),
                                status_code=status.HTTP_200_OK)
        # checking for the account and password matching
        user_data_response, data = self.obj_login_handler.db_password_matching(self.login_type, user_data,
                                                                               decrypted_password.split("\"")[1])
        # if the passwords doesn't match with the db data
        if user_data_response is not None:
            return JSONResponse(content=DefaultFailureResponse(status="failed",
                                                               message=data).dict(),
                                status_code=status.HTTP_200_OK)
        #  generating the access tokens
        responses, exp = self.obj_login_handler.generate_cookie_tokens(data, request)
        # token generation unsuccessful
        if responses is None:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_TOKEN_GENERATION).dict(),
                status_code=status.HTTP_200_OK)
        # sending successful response to UI
        response = JSONResponse(
            content=DefaultResponse(status="success", message="Logged In Successfully",
                                    data=data).dict(),
            status_code=status.HTTP_200_OK, headers={"Content-Type": "application/json"})
        response.set_cookie(key="login-token", value=responses, expires=exp)
        return response

    def google_login(self, user_data, request):
        user_data_remove_none = {key: value for key, value in user_data if key != 'login_type' and value is not None}
        req = requests.Request()
        try:
            id_info = id_token.verify_oauth2_token(
                user_data_remove_none["id_token"], req, Secrets.CLIENT_ID)
        except InvalidValue:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_TOKEN_EXPIRED).dict(),
                status_code=status.HTTP_200_OK)
        response, message = self.obj_login_handler.db_data_validation(user_data.login_type, id_info["email"])
        # if the response is false then an error message is send back
        if response is not None:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=message).dict(),
                status_code=status.HTTP_200_OK)
        message.update({"name": id_info["name"], "pic_url": id_info["picture"]})
        responses = self.obj_login_handler.update_pic(obj_mongo_user, id_info)
        if responses is None:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_IN_UPDATING).dict(),
                status_code=status.HTTP_200_OK)
        # generating the access tokens
        responses, exp = self.obj_login_handler.generate_cookie_tokens(message, request)
        # token generation unsuccessful
        if responses is None:
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_TOKEN_GENERATION).dict(),
                status_code=status.HTTP_200_OK)
            # sending successful response to UI
        response = JSONResponse(
            content=DefaultResponse(status="success", message="Logged In Successfully",
                                    data=message).dict(),
            status_code=status.HTTP_200_OK, headers={"Content-Type": "application/json"})
        response.set_cookie(key="login-token", value=responses, expires=exp)
        return response

    # v2
    def microsoft_login(self, request):
        pass

    # forgot password handler
    @staticmethod
    def forgot_password_handler(email):
        try:
            # Check if user exists in database
            # If user exists, send forgot password email with JWT token
            # This should include email and expire time
            # Send email using MIME
            db_user_data = obj_mongo_user.fetch_one_user_details({"email": email})
            # if the user is not available
            if not db_user_data:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_USER_ID_DOESNT_EXIST).dict(),
                    status_code=status.HTTP_200_OK)
            mail = MIMEMultipart()
            mail['From'] = Services.EMAIL_SENDER
            mail['To'] = email
            mail['Subject'] = "Link To Reset Password"
            to_encode = {"email": email}
            expire = datetime.utcnow() + timedelta(minutes=Secrets.TOKEN_EXPIRE_TIME)
            to_encode.update({"exp": expire})
            jwt_token = jwt.encode(to_encode)
            html = ''
            # Load the HTML file
            try:
                with open(Services.HTML_LINK, "r") as f:
                    html = f.read()
                html = html.replace("{{ message }}", "Please click the link to reset your password:").replace(
                    "{{ link }}", Services.RESET_ENDPOINT + "=" + str(jwt_token))
            except Exception as e:
                logger.exception(e)
            html_body = MIMEText(html, "html")
            mail.attach(html_body)
            context = ssl.create_default_context()
            with smtplib.SMTP_SSL(Services.EMAIL_SMTP, Services.EMAIL_PORT, context=context) as smtp:
                smtp.login(Services.EMAIL_SENDER, Services.EMAIL_PASSWORD)
                # sending the mail
                smtp.sendmail(Services.EMAIL_SENDER, email, mail.as_string())

            return JSONResponse(
                content=DefaultSuccessResponse(status="success", message="Email Send Successfully").dict(),
                status_code=status.HTTP_200_OK)
        except Exception as e:
            logger.exception(e)

    @staticmethod
    def validate_jwt(request):
        try:
            jwt_token = request.query_params['token']
            jwt_token_encoded = jwt_token.encode('utf-8')
            # Verify and decode the JWT token
            try:
                decoded_token = jwt.decode(jwt_token_encoded)
                db_user_data = obj_mongo_user.fetch_one_user_details({"email": decoded_token['email']})
                # if the user is not available
                if not db_user_data:
                    return JSONResponse(
                        content=DefaultFailureResponse(status="failed",
                                                       message=ErrorMessages.ERROR_USER_ID_DOESNT_EXIST).dict(),
                        status_code=status.HTTP_200_OK)
                jwt_token_new = jwt.encode({"email": decoded_token['email']})
                return RedirectResponse('http://192.168.2.102/iLens_UI/#/l/login?user_id=' + jwt_token_new)
            except ExpiredSignatureError:
                return RedirectResponse(
                    'http://192.168.2.102/iLens_UI/#/l/login?error=' + "true")
        except Exception as e:
            logger.exception(e)

    @staticmethod
    def reset_user_password(reset_data):
        try:
            user_id_token = reset_data.user_id
            user_id = user_id_token.encode('utf-8')
            # Verify and decode the JWT token
            try:
                decoded_token = jwt.decode(user_id)
                db_user_data = obj_mongo_user.fetch_one_user_details({"email": decoded_token['email']})
                # if the user is not available
                if not db_user_data:
                    return JSONResponse(
                        content=DefaultFailureResponse(status="failed",
                                                       message=ErrorMessages.ERROR_USER_ID_DOESNT_EXIST).dict(),
                        status_code=status.HTTP_200_OK)
                response = EncryptDecryptPassword().password_encrypt(reset_data.password)
                if not response:
                    return JSONResponse(
                        content=DefaultFailureResponse(status="failed",
                                                       message=ErrorMessages.ERROR_AUTH_FAILED).dict(),
                        status_code=status.HTTP_200_OK)
                response = obj_mongo_user.update_user({"email": decoded_token['email']}, {"password": response})
                if not response:
                    return JSONResponse(
                        content=DefaultFailureResponse(status="failed",
                                                       message=ErrorMessages.ERROR_IN_UPDATING).dict(),
                        status_code=status.HTTP_200_OK)
                return JSONResponse(
                    content=DefaultSuccessResponse(status="success", message="Reset Successful").dict(),
                    status_code=status.HTTP_200_OK)
            except ExpiredSignatureError:
                return "Password Reset Token Expired"
        except Exception as e:
            logger.exception(e)
