import datetime
import uuid

from passlib.context import CryptContext

from scripts.database.mongo.mongo_db import MongoUser
from scripts.database.redis.redis_conn import login_db
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger
from fastapi.responses import JSONResponse
from fastapi import status
from scripts.schemas.default_responses import DefaultResponse, DefaultFailureResponse, DefaultSuccessResponse
from scripts.utils.mongo_utils import MongoStageCreator
from scripts.utils.response_utils import ResponseData
from scripts.utils.security.password_util import EncryptDecryptPassword
from scripts.utils.validations_util import UserDataValidations

obj_mongo_user = MongoUser()
obj_response_data = ResponseData()
obj_stage = MongoStageCreator()


# user management
class UserManagement:
    def __init__(self):
        self.method = "register"
        self.pass_decrypt = EncryptDecryptPassword()
        self.pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")

    # for normal registration using email and password
    def general_register(self, user_data):
        try:
            response, message = UserDataValidations.register_data_validation(user_data, 'general', self.method)
            if not response:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=message["message"]).dict(),
                    status_code=status.HTTP_200_OK)
            # fetching the data based on the username
            db_user_data = obj_mongo_user.fetch_one_user_details({"email": user_data.email})
            # if the user is not available
            if db_user_data:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_EMAIL_EXIST).dict(),
                    status_code=status.HTTP_200_OK)
            # creating a unique user id
            uid = str(uuid.uuid4()).replace("-", "")
            user_data.user_id = "user_" + uid
            created_at = datetime.datetime.now()
            updated_at = datetime.datetime.now()
            reg_time = {"created_at": created_at, "updated_at": updated_at}
            encrypted = EncryptDecryptPassword().password_encrypt(user_data.password)
            if encrypted is None:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_INVALID_PASSWORD).dict(),
                    status_code=status.HTTP_200_OK)
            # storing the user data in dict, encrypting the password
            user_data_dict = {key: (encrypted
                                    if key == "password" else value) for key, value in user_data if key != 'action'}
            # adding the registration time to the user data
            user_data_reg = user_data_dict | reg_time
            # checking if the insertion is working
            if not obj_mongo_user.insert_new_user(user_data_reg):
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_STORING_DATA).dict(),
                    status_code=status.HTTP_200_OK)
            return JSONResponse(
                content=DefaultSuccessResponse(status="success", message="User Registration Successful").dict(),
                status_code=status.HTTP_200_OK)
        except Exception as e:
            logger.exception(f'Services Failed with error from general register {e}')
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_IN_REGISTERING).dict(),
                status_code=status.HTTP_200_OK)

    @staticmethod
    # for Google registration using gmail
    def google_register(user_data):
        try:
            # fetching the data based on the username
            db_user_data = obj_mongo_user.fetch_one_user_details({"email": user_data.email})
            # if the user is not available
            if db_user_data:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_EMAIL_EXIST).dict(),
                    status_code=status.HTTP_200_OK)
            # creating a unique user id
            uid = str(uuid.uuid4()).replace("-", "")
            user_data.user_id = "user_" + uid
            created_at = datetime.datetime.now()
            updated_at = datetime.datetime.now()
            reg_time = {"created_at": created_at, "updated_at": updated_at}
            # removing action and none values of the user data
            user_data_dict = {key: value for key, value in user_data if key != 'action' and value is not None}
            user_data_reg = user_data_dict | reg_time
            # checking if the insertion is working
            if not obj_mongo_user.insert_new_user(user_data_reg):
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_STORING_DATA).dict(),
                    status_code=status.HTTP_200_OK)
            return JSONResponse(
                content=DefaultSuccessResponse(status="success", message="User Registration Successful").dict(),
                status_code=status.HTTP_200_OK)
        except Exception as e:
            logger.exception(f'Services Failed with error from google register {e}')
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_IN_REGISTERING).dict(),
                status_code=status.HTTP_200_OK)

    @staticmethod
    # for microsoft registration using microsoft account
    def microsoft_register():
        try:
            return {"message": "Not available"}
        except Exception as e:
            logger.exception(f'Services Failed with error from microsoft register {e}')
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_IN_REGISTERING).dict(),
                status_code=status.HTTP_200_OK)

    # update user details
    def update_user_details(self, update_data):
        try:
            self.method = "update"
            # fetching and validating the user details
            db_user_data = obj_mongo_user.fetch_one_user_details({"user_id": update_data.user_id})
            # if the user is not available
            if db_user_data is None:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_USER_ID_DOESNT_EXIST).dict(),
                    status_code=status.HTTP_404_NOT_FOUND)
            # checking the user email and validating
            if update_data.email is not None:
                db_user_data = obj_mongo_user.fetch_one_user_details(update_data.email)
                # if the user is not available
                if db_user_data is not None:
                    return JSONResponse(
                        content=DefaultFailureResponse(status="failed",
                                                       message=ErrorMessages.ERROR_EMAIL_EXIST).dict(),
                        status_code=status.HTTP_404_NOT_FOUND)
            # creating the filter data
            filter_data_updated = {"user_id": update_data.user_id}
            encrypted = EncryptDecryptPassword().password_encrypt(update_data.password)
            if encrypted is None:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_INVALID_PASSWORD).dict(),
                    status_code=status.HTTP_200_OK)
            # encrypting the password
            update_data_removed = {key: (encrypted
                                         if key == "password" else value) for key, value in update_data if
                                   key != 'action' and value is not None}
            # validating the data
            response, message = UserDataValidations.update_data_validation(update_data)
            if not response:
                return message
            # updating the data
            response = obj_mongo_user.update_user(filter_data_updated, update_data_removed)
            if not response:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_IN_UPDATING).dict(),
                    status_code=status.HTTP_200_OK)
            return JSONResponse(
                content=DefaultSuccessResponse(status="success", message="Updated Successfully").dict(),
                status_code=status.HTTP_200_OK)
        except Exception as e:
            logger.exception(f'Services Failed with error from update user {e}')
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_IN_UPDATING).dict(),
                status_code=status.HTTP_200_OK)

    # delete user
    @staticmethod
    def delete_user_details(user_id):
        try:
            # fetching and validating the user id
            db_user_data = obj_mongo_user.fetch_one_user_details({"user_id": user_id})
            # if the user is not available
            if db_user_data is None:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_USER_ID_DOESNT_EXIST).dict(),
                    status_code=status.HTTP_404_NOT_FOUND)
            # generating the filter
            filter_data_updated = {"user_id": user_id}
            # deleting the user
            response = obj_mongo_user.delete_user(filter_data_updated)
            if not response:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_IN_UPDATING).dict(),
                    status_code=status.HTTP_404_NOT_FOUND)
            return JSONResponse(
                content=DefaultSuccessResponse(status="success", message="Deleted Successfully").dict(),
                status_code=status.HTTP_200_OK)
        except Exception as e:
            logger.exception(f'Services Failed with error from delete user {e}')

    @staticmethod
    def fetch_view_header():
        try:
            # fetching the table header for the user view
            data_response = obj_response_data.user_view_header()
            if data_response:
                return JSONResponse(
                    content=DefaultResponse(status="success", message="Header Fetched Successfully",
                                            data=data_response).dict(),
                    status_code=status.HTTP_200_OK)
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_IN_FETCHING).dict(),
                status_code=status.HTTP_404_NOT_FOUND)
        except Exception as e:
            logger.exception(f'Services Failed with error from fetch user header {e}')
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_IN_FETCHING_HEADER).dict(),
                status_code=status.HTTP_200_OK)

    @staticmethod
    def fetch_user_details():
        try:
            # defining the filter values
            filter_data = {'_id': 0,
                           "login_type": 0,
                           "is_alive": 0,
                           "password": 0,
                           "created_at": 0,
                           "updated_at": 0}
            # filtering the users and getting all the details
            cursor_data = obj_mongo_user.fetch_all_user_details({}, filter_data)
            cursor_data_count = cursor_data.explain()
            # counting the total records in the query
            if cursor_data_count["executionStats"]["nReturned"] <= 0:
                return None
            list_user_data = []
            for users in cursor_data:
                list_user_data.append(users)
            # listing the user data
            if list_user_data:
                return JSONResponse(
                    content=DefaultResponse(status="success", message="Fetched Successfully",
                                            data={"bodyContent": list_user_data}).dict(),
                    status_code=status.HTTP_200_OK)
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_IN_FETCHING).dict(),
                status_code=status.HTTP_404_NOT_FOUND)
        except Exception as e:
            logger.exception(f'Services Failed with error from fetch user details {e}')
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_IN_FETCHING).dict(),
                status_code=status.HTTP_200_OK)

    # user change password
    def reset_password(self, reset_data):
        try:
            db_user_data = obj_mongo_user.fetch_one_user_details({"user_id": reset_data.user_id})
            # if the user is not available
            if db_user_data is None:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_USER_ID_DOESNT_EXIST).dict(),
                    status_code=status.HTTP_404_NOT_FOUND)
            try:
                decrypted_password = self.pass_decrypt.password_decrypt(reset_data.new_password)
            except TypeError:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_INVALID_PASSWORD).dict(),
                    status_code=status.HTTP_200_OK)
            if not self.pwd_context.verify(decrypted_password.split("\"")[1], db_user_data["password"]):
                return JSONResponse(
                            content=DefaultFailureResponse(status="failed",
                                                           message=ErrorMessages.ERROR_PASSWORD_MISMATCH).dict(),
                            status_code=status.HTTP_200_OK)
            response = EncryptDecryptPassword().password_encrypt(reset_data.new_password)
            if not response:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_AUTH_FAILED).dict(),
                    status_code=status.HTTP_200_OK)
            response = obj_mongo_user.update_user({"user_id": reset_data.user_id}, {"password": response})
            if not response:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_IN_UPDATING).dict(),
                    status_code=status.HTTP_200_OK)
            return JSONResponse(
                content=DefaultSuccessResponse(status="success", message="Password Changed Successfully").dict(),
                status_code=status.HTTP_200_OK)
        except Exception as e:
            logger.exception(f'Services Failed with error from reset user password {e}')
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_IN_RESETTING_PASSWORD).dict(),
                status_code=status.HTTP_200_OK)

    # user logout
    @staticmethod
    def logout_user(user_id, request):
        try:
            # getting the cookie token
            uid = request.login_token
            # checking the user id with the user id stored in cookie
            if user_id.user_id != request.user_id:
                return JSONResponse(
                    content=DefaultFailureResponse(status="failed",
                                                   message=ErrorMessages.ERROR_USER_SESSION).dict(),
                    status_code=status.HTTP_404_NOT_FOUND)
            # deleting the login token from redis
            login_db.delete(uid)
            response = JSONResponse(
                content=DefaultSuccessResponse(status="success", message="Logged Out").dict(),
                status_code=status.HTTP_200_OK)
            # deleting the cookie
            response.delete_cookie("login-token")
            return response
        except Exception as e:
            logger.exception(f'Services Failed with error from user logout {e}')
            return JSONResponse(
                content=DefaultFailureResponse(status="failed",
                                               message=ErrorMessages.ERROR_IN_LOGOUT).dict(),
                status_code=status.HTTP_200_OK)
