from __future__ import annotations

from datetime import datetime

from passlib.context import CryptContext
from validate_email import validate_email

from scripts.database.mongo.mongo_db import MongoUser
from scripts.errors import ErrorMessages
from scripts.logging.logger import logger
from scripts.utils.security.apply_encrytion_util import create_token


class NormalLogin:
    def __init__(self):
        self.pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")
        self.db_user_data = None
        self.db_user_data = None
        self.dt = datetime.now()
        self.time_dt = datetime.now()

    # user data validation
    @staticmethod
    def user_data_validation(email, password) -> dict | None:
        try:
            # checking for valid username
            if email == "" or validate_email(email) is not True:
                return ErrorMessages.ERROR_INVALID_EMAIL
            # checking for valid password
            if password == "":
                return ErrorMessages.ERROR_INVALID_PASSWORD
            return None
        except Exception as e:
            logger.error(f'Services Failed with error from process login user data validation {e}')
            return None

    # db validation
    def db_data_validation(self, login_type, email):
        try:
            # fetching the data based on the username
            self.db_user_data = MongoUser().fetch_one_user_details({"email": email})
            # if the user is not available
            if not self.db_user_data:
                return False, ErrorMessages.ERROR_ACCOUNT_DOESNT_EXIST
            # if the user is not registered through normal login
            if self.db_user_data["login_type"] != login_type:
                return False, ErrorMessages.ERROR_ACCESS_DENIED
            try:
                response = {"user_id": self.db_user_data["user_id"], "name": self.db_user_data["name"],
                            "email": email,
                            "user_role": self.db_user_data["user_role"]}
            except KeyError:
                response = {"user_id": self.db_user_data["user_id"],
                            "email": email,
                            "user_role": self.db_user_data["user_role"]}
            # if the user exist
            return None, response
        except Exception as e:
            logger.error(f'Services Failed with error from process login db data validation {e}')
            return True, "Exception"

    # matching the password
    def db_password_matching(self, login_type, user_data, password):
        try:
            # getting the response after checking for the user data in db
            response, message = self.db_data_validation(login_type, user_data.email)
            # if the response is false then an error message is send back
            if response is not None:
                return response, message
            # if the user exists in db then password is matched
            if not self.pwd_context.verify(password, self.db_user_data["password"]):
                return False, ErrorMessages.ERROR_PASSWORD_MISMATCH
            # if the password is correct
            return None, message
        except Exception as e:
            logger.error(f'Services Failed with error from process login db password matching {e}')
            return True, "Exception"

    @staticmethod
    def update_pic(obj_mongo_user, info_data):
        try:
            if not obj_mongo_user.update_user({"email": info_data["email"]},
                                              {"name": info_data["name"], "pic_url": info_data["picture"]}):
                return None
            return True
        except Exception as e:
            logger.error(f'Services Failed with error from process login update pic {e}')
            return None

    # cookie and token creation
    @staticmethod
    def generate_cookie_tokens(user_data, request):
        try:
            # creating the access token
            access_token, exp = create_token(
                user_id=user_data["user_id"],
                ip=request.client.host
            )
            # returning the login token
            if access_token:
                return access_token, exp
            else:
                return None
        except Exception as e:
            logger.error(f'Services Failed with error from process login cookie tokens {e}')
            return None
