from abc import ABC, abstractmethod
from math import pi


# Abstract factory interface
class AbstractShapeFactory(ABC):
    @abstractmethod
    def create_shape(self):
        pass


# Concrete factory for creating circles
class CircleFactory(AbstractShapeFactory):
    def create_shape(self, radius: float):
        return Circle(radius)


# Concrete factory for creating rectangles
class RectangleFactory(AbstractShapeFactory):
    def create_shape(self, width: float, height: float):
        return Rectangle(width, height)


# Abstract product interface
class AbstractShape(ABC):
    @abstractmethod
    def area(self):
        pass


# Concrete product class for circles
class Circle(AbstractShape):
    def __init__(self, radius: float):
        self.radius = radius

    def area(self):
        return pi * self.radius ** 2


# Concrete product class for rectangles
class Rectangle(AbstractShape):
    def __init__(self, width: float, height: float):
        self.width = width
        self.height = height

    def area(self):
        return self.width * self.height


# Client code that uses the abstract factory
class AreaCalculator:
    def __init__(self, factory: AbstractShapeFactory, *args):
        self.shape = factory.create_shape(*args)

    def calculate_area(self):
        return self.shape.area()
