import logging
import os
import pathlib
from logging import StreamHandler
from logging.handlers import RotatingFileHandler, SocketHandler

from scripts.config import LoggerConf, PathToStorage, Service


def read_configuration():
    return {
        "name": "ut_module",
        "handlers": [
            {
                "type": "RotatingFileHandler",
                "max_bytes": 100000000,
                "back_up_count": 5,
                "enable": False
                if os.environ.get("ENABLE_FILE_LOG", False) in [False, "False", "false"]
                else True,
            },
            {
                "type": "StreamHandler",
                "enable": False
                if os.environ.get("ENABLE_CONSOLE_LOG", True)
                in [False, "False", "false"]
                else True,
            },
        ],
    }


logging_config = read_configuration()


def get_logger():
    """
    Creates a rotating log
    """
    __logger__ = logging.getLogger("")
    _level = LoggerConf.logging_level.upper()
    if not _level:
        _level = "INFO"
    __logger__.setLevel(_level)
    log_formatter = "%(asctime)s - %(levelname)-6s - " \
                    "[%(threadName)5s:%(funcName)5s(): %(lineno)s] - %(message)s"
    time_format = "%Y-%m-%d %H:%M:%S"
    formatter = logging.Formatter(log_formatter, time_format)
    for each_handler in logging_config["handlers"]:
        if each_handler["type"] in ["RotatingFileHandler"] and each_handler.get(
            "enable", False
        ):
            pathlib.Path(PathToStorage.LOG_PATH).mkdir(parents=True, exist_ok=True)
            log_file = pathlib.Path(
                PathToStorage.LOG_PATH, f"{Service.module_name}.log"
            )
            __logger__.debug("File Logger Enabled")
            temp_handler = RotatingFileHandler(
                log_file,
                maxBytes=each_handler["max_bytes"],
                backupCount=each_handler["back_up_count"],
            )
            temp_handler.setFormatter(formatter)
        elif each_handler["type"] in ["StreamHandler"] and each_handler.get(
            "enable", True
        ):
            __logger__.debug("Console Logger Enabled")
            temp_handler = StreamHandler()
            temp_handler.setFormatter(formatter)
        elif each_handler["type"] in ["SocketHandler"]:
            temp_handler = SocketHandler(each_handler["host"], each_handler["port"])
        else:
            continue
        __logger__.addHandler(temp_handler)

    return __logger__


logger = get_logger()
