"""
This is main script where fastapi application is initialized.

Running this file directly should be avoided, it is ideal to run app.py or debug.py
"""
from dotenv import load_dotenv

from scripts.constants.secrets import Secrets
from scripts.services.ut_service import ut_router

load_dotenv()

import gc
import os

from __version__ import version

from fastapi import APIRouter, Depends, FastAPI
from fastapi.middleware.cors import CORSMiddleware
from jwt_signature_validator.encoded_payload import (
    EncodedPayloadSignatureMiddleware as SignatureVerificationMiddleware,
)

from scripts.config import Service, Security

from scripts.utils.security_utils.decorators import CookieAuthentication

secure_access = os.environ.get("SECURE_ACCESS", default=False)
auth = CookieAuthentication()
gc.collect()

router = APIRouter(tags=["ping"])


@router.get("/api/ut_module/healthcheck")
async def ping():
    return {"status": 200}


app = FastAPI(
    title="UT",
    version=version,
    description="UT module",
    root_path=Service.PROXY,
    openapi_url=os.environ.get("SW_OPENAPI_URL"),
    docs_url=os.environ.get("SW_DOCS_URL"),
    redoc_url=None,
)

# Signature Verification
if Security.verify_signature:
    app.add_middleware(
        SignatureVerificationMiddleware,
        jwt_secret=Secrets.signature_key,
        jwt_algorithms=Secrets.signature_key_alg,
        protect_hosts=Service.protected_hosts,
    )


app.include_router(ut_router)


if os.environ.get("ENABLE_CORS") in (True, "true", "True") and os.environ.get(
    "CORS_URLS"
):
    app.add_middleware(
        CORSMiddleware,
        allow_origins=os.environ.get("CORS_URLS").split(","),
        allow_credentials=True,
        allow_methods=["GET", "POST", "DELETE", "PUT"],
        allow_headers=["*"],
    )
