import os.path
import sys

from pydantic import BaseSettings, Field


class _Service(BaseSettings):
    PORT: int = 9876
    HOST: str = "0.0.0.0"
    PROXY: str = Field(..., env="MODULE_PROXY")
    module_name = Field(default="ut_module", env="MODULE_NAME")


class _Security(BaseSettings):
    secure_cookie: bool = os.environ.get("SECURE_COOKIE", default=True)
    verify_signature: bool = os.environ.get("VERIFY_SIGNATURE", default=False)
    protected_hosts: list = os.environ.get("PROTECTED_HOSTS", default="").split(",")


class _MongoConf(BaseSettings):
    uri: str = Field(..., env="MONGO_URI")


class _RedisConf(BaseSettings):
    uri: str = Field(..., env="REDIS_URI")
    login_db: int = Field(default=9, env="REDIS_LOGIN_DB")
    project_tags_db: int = Field(default=18, env="REDIS_PROJECT_DB")
    user_role_permissions: int = Field(default=21, env="REDIS_USER_ROLE_DB")


class _MQTTConf(BaseSettings):
    uri: str = Field(env="MQTT_URI")
    host: str = Field(..., env="MQTT_HOST")
    port: int = Field(..., env="MQTT_PORT")
    publish_base_topic = Field(default="ilens/notifications")


class _PathToStorage(BaseSettings):
    BASE_PATH: str = os.environ.get("BASE_PATH")
    MOUNT_DIR: str = os.environ.get("MOUNT_DIR")
    if not any([BASE_PATH, MOUNT_DIR]):
        print("Error, environment variable BASE_PATH or  MOUNT_DIRnot set")
        sys.exit(1)
    MODULE_PATH: str = os.path.join(BASE_PATH, MOUNT_DIR)
    IMAGES_PATH: str = os.path.join(MODULE_PATH, "images")
    LOG_PATH: str = os.path.join(BASE_PATH, "logs", MOUNT_DIR)


class _LoggerConf(object):
    logging_level: str = os.environ.get("LOG_LEVEL", default="DEBUG")


class _KeyPath(BaseSettings):
    keys_path: str = "keys"
    public: str = os.path.join(_PathToStorage().BASE_PATH, keys_path, "public")
    private: str = os.path.join(_PathToStorage().BASE_PATH, keys_path, "private")


Service = _Service()
Security = _Security()
MongoConf = _MongoConf()
RedisConf = _RedisConf()
MQTTConf = _MQTTConf()
PathToStorage = _PathToStorage()
LoggerConf = _LoggerConf()
KeyPath = _KeyPath()

__all__ = [
    "Service",
    "Security",
    "MongoConf",
    "RedisConf",
    "MQTTConf",
    "PathToStorage",
    "LoggerConf",
    "KeyPath",
]
