import threading
from datetime import datetime

import uvicorn
from fastapi import FastAPI

from scripts.constants.api_endpoints import api_endpoints
from scripts.constants.app_variables import TimeVariable
from scripts.constants.app_config import app_config
from scripts.core.handlers.device_handler import DeviceHandler
from scripts.core.handlers.files_handler import FilesHandler
from scripts.core.handlers.redundancy_handler import RedundancyHandler
from scripts.logging.logger import get_logger

logger = get_logger("secondary_redundancy")

app = FastAPI()

files_handler = FilesHandler(logger)
redundancy_handler = RedundancyHandler(logger)
device_handler = DeviceHandler(logger)

api_hit_thread = threading.Thread(target=device_handler.check_last_hit)

api_hit_thread.start()


@app.post(api_endpoints.receive_data)
async def receive_data(data: dict):
    response = {
        "status": "Failed",
        "message": "File updating failed."
    }
    try:
        given_path = data.get("file_name")
        file_path = files_handler.is_file_path_need(given_path)
        config_data = data.get("data")
        files_handler.sync_config(file_path, config_data)
        logger.info("File updated successfully.")
        response["status"] = "Success"
        response["message"] = "File updating successful."
    except Exception as e:
        logger.Error(f"Exception occurred while updating file - {e}.")
    return response


@app.post(api_endpoints.config_sync)
async def receive_config(data: dict):
    response = {
        "status": "Failed",
        "message": "Configuration syncing failed."
    }
    try:
        for given_path, config_data in data.items():
            file_path = files_handler.is_file_path_need(given_path)
            files_handler.sync_config(file_path, config_data)
        logger.info("Configuration synced successfully.")
        response["status"] = "Success"
        response["message"] = "Configuration syncing successful."
    except Exception as e:
        logger.error(f"Exception occurred while performing configuration syncing - {e}.")
    return response


@app.post(api_endpoints.fetch_details)
async def fetch_primary_device_details(data: dict):
    response = {
        "status": "Failed",
        "message": "Primary device details fetching failed.",
        "action": None
    }
    try:
        response_data = f"Heartbeat message - {data}"
        logger.info(response_data)
        logger.info(f"TimeVariable.api_last_hit_time in receiving hb entry- {TimeVariable.api_last_hit_time}")
        TimeVariable.api_last_hit_time = datetime.now()
        logger.info(
            f"TimeVariable.api_last_hit_time in receiving hb after updating time- {TimeVariable.api_last_hit_time}")
        device_handler.check_system_time(data.get("date_time"))
        device_handler.check_last_quality_data_time(data.get("data_check"))
        response["status"] = "Success"
        response["message"] = "Primary device details fetching successful."
        response["action"] = redundancy_handler.response_action(data.get("data_check"))
    except Exception as e:
        logger.error(f"Exception occurred while fetching primary device details - {e}.")
    return response


if __name__ == "__main__":
    try:
        logger.info("-----------------------------------------Starting Service-----------------------------------------")
        uvicorn.run(app, host=app_config.host_ip, port=app_config.port_no)
    except KeyboardInterrupt:
        logger.info("---------------------------------------Service Stopped---------------------------------------")
