import subprocess
import time

from scripts.constants.app_constants import constants, services
from scripts.logging.logger import logger


class ServiceOperations:
    @staticmethod
    def is_service_enabled(service_name):
        try:
            logger.trace("Entering in is_service_enabled")
            output = subprocess.check_output(["sc", "query", service_name], universal_newlines=True)
            if "STATE" in output and "START_TYPE" in output:
                state_line = None
                start_type_line = None
                for line in output.splitlines():
                    if "STATE" in line:
                        state_line = line
                    elif "START_TYPE" in line:
                        start_type_line = line
                state = state_line.split(":")[1].strip()
                start_type = start_type_line.split(":")[1].strip()
                return state == "4" and start_type == "2"
        except subprocess.CalledProcessError:
            pass
        logger.trace("Exiting from is_service_enabled")
        return False

    def restart_service(self, service_name):
        try:
            logger.trace("Entering in restart_service")
            if constants.is_windows:
                if not self.is_service_enabled(service_name):
                    subprocess.run(["sc", "config", service_name, "start=auto"], check=True)
                    time.sleep(5)
                current_status = subprocess.check_output(["sc", "query", service_name]).decode("utf-8")
                time.sleep(5)
                if "RUNNING" in current_status:
                    subprocess.run(["sc", "stop", service_name], check=True)
                    time.sleep(5)
                subprocess.run(["sc", "start", service_name], check=True)
                time.sleep(5)
                return True
            else:
                subprocess.run(["sudo", "systemctl", "restart", service_name], check=True)
                result = subprocess.run(
                    ["sudo", "systemctl", "is-active", service_name],
                    capture_output=True,
                    text=True,
                )
                if result.returncode == 0:
                    return True
                else:
                    return False
        except subprocess.CalledProcessError as e:
            logger.exception(f"Failed to restart the service '{service_name}'. Error: {e}")
        logger.trace("Exiting from restart_service")
        return False

    def stop_service(self, service_name):
        try:
            logger.trace("Entering in stop_service")
            if constants.is_windows:
                subprocess.run(["sc", "stop", service_name], check=True)
                if self.is_service_enabled(service_name):
                    subprocess.run(["sc", "config", service_name, "start=disabled"], check=True)
                time.sleep(5)
                logger.trace("Exiting from stop_service in windows as true")
                return True
            else:
                subprocess.run(["sudo", "systemctl", "disable", service_name], check=True)
                subprocess.run(["sudo", "systemctl", "stop", service_name], check=True)
                result = subprocess.run(
                    ["sudo", "systemctl", "is-active", service_name],
                    capture_output=True,
                    text=True,
                )
                if result.returncode == 0:
                    logger.trace("Exiting from stop_service in linux as false - service doesn't stopped")
                    return False
                else:
                    logger.trace("Exiting from stop_service in linux as true - service stopped")
                    return True
        except Exception as e:
            logger.exception(f"Exception occurred while stopping {service_name} - {e}.")
        logger.trace("Exiting from stop_service")
        return False

    @staticmethod
    def check_ilens_client_status(logger):
        try:
            logger.trace("Entering in check_ilens_client_status")
            if constants.is_windows:
                output = subprocess.check_output(
                    ["sc", "query", services.acquisition_engine], universal_newlines=True
                )
                if "STATE" in output:
                    for line in output.splitlines():
                        if "STATE" in line and "RUNNING" in line:
                            logger.trace("Exiting from check_ilens_client_status in windows as true - service is active.")
                            return True
                logger.trace("Exiting from check_ilens_client_status in windows as false - service is inactive")
                return False
            else:
                result = subprocess.run(
                    ["sudo", "systemctl", "is-active", services.acquisition_engine],
                    capture_output=True,
                    text=True,
                )
                if result.returncode == 0:
                    logger.trace("Exiting from check_ilens_client_status in linux as true - service is active")
                    return True
                else:
                    logger.trace("Exiting from check_ilens_client_status in linux as false - service is inactive")
                return False
        except Exception as e:
            logger.exception(f"Exception occurred while check_ilens_client_status - {e}")

    @staticmethod
    def service_name_mapper(file_path):
        try:
            logger.trace(f"Entering in service_name_mapper with filepath - {file_path}.")
            service_modules = {
                ".conf": services.ilens_agent,
                "monitoring_engine": services.monitoring_engine,
                "ilens_heartbeat": services.ilens_heartbeat,
                "ilens_data_receiver": services.ilens_data_receiver,
                "ilens_data_transmitter": services.ilens_data_transmitter,
                "ilens_events_transmitter": services.ilens_events_transmitter,
                "ilens_file_transmitter": services.ilens_file_transmitter
            }
            for module_keyword, module_service in service_modules.items():
                if module_keyword in file_path:
                    logger.trace(f"Exiting from service_name_mapper for {module_service} module.")
                    return module_service
        except Exception as e:
            logger.exception(f"Exception occurred while mapping service name for module - {e}")
        return None


service_operations = ServiceOperations()
