from datetime import datetime

from scripts.constants.app_constants import constants
from scripts.constants.app_variables import DataPool
from scripts.constants.events import events_constants
from scripts.constants.app_config import app_config
from scripts.logging.logger import logger
from scripts.utilities.common_util import common_utilities
from scripts.utilities.communication_util import post_events
from scripts.utilities.service_util import service_operations


class RedundancyHandler:
    def __init__(self, logger):
        self.logger = logger

    def fetch_pipeline_details(self, pipeline_data):
        self.logger.trace("Entered in fetch_pipeline_details")
        pipeline_version = "upgrade"
        try:
            pipeline_version = pipeline_data.get("pipeline_version")
        except Exception as e:
            self.logger.exception(f"No channel pipeline file found - {e}.")
        self.logger.trace(f"Exiting from fetch_pipeline_details with pipeline_version - {pipeline_version}")
        return pipeline_version

    @staticmethod
    def read_current_time():
        date_time = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        return date_time

    def check_system_architecture(self, pipeline_data):
        connection_status = "NA"
        self.logger.trace("Entered in check_system_architecture")
        try:
            flow_data_node_list = [
                node for node in pipeline_data.get("flow_data", {}).get("nodes", {}).keys()
                if not node.startswith("device_instance_") if not node.startswith("collector_")
            ]
            system_dependency = "No dependency"
            for node in flow_data_node_list:
                if node.startswith("udp") and pipeline_data.get("flow_data").get("nodes").get(node).get(
                        "node_configuration").get("host") not in ["127.0.0.1", "localhost"]:
                    self.logger.trace("Have diode dependency")
                    system_dependency = "Diode dependency"
            node_configuration_details = pipeline_data.get("flow_data").get(
                "nodes").get(flow_data_node_list[0]).get("node_configuration")
            uploader_name = node_configuration_details.get("name")
            host_ip = node_configuration_details.get("host")
            host_port = node_configuration_details.get("port")
            host_status = common_utilities.ping_host(host_ip)
            connection_status = {
                "system_dependency": system_dependency,
                "uploader": uploader_name,
                "host_ip": host_ip,
                "host_port": host_port,
                "host_status": "Active" if host_status else "Inactive"
            }

        except Exception as e:
            self.logger.exception(f"Exception occurred while checking system architecture - {e}.")
        self.logger.trace("Exiting from check_system_architecture")
        return connection_status

    def fetch_device_details(self):
        response = None
        self.logger.trace("Entered in fetch_device_details")
        try:
            try:
                data_check = False
                if DataPool.data_pool:
                    self.logger.trace(f"data pool: \'{DataPool.data_pool}\'")
                    data_quality_set = set(DataPool.data_pool)
                    data_check = True
                    if len(data_quality_set) == 1 and list(data_quality_set)[0] == 2:
                        data_check = False
                        post_events(events_constants.bad_data_quality)
            except Exception as e:
                self.logger.error(f"data_check exception: {e}")
                data_check = False
            pipeline_data = common_utilities.read_configs(constants.channel_pipeline_path)
            response = {
                "pipeline_version": self.fetch_pipeline_details(pipeline_data) if app_config.is_data_source else "NA",
                "date_time": self.read_current_time(),
                "connection_status": (self.check_system_architecture(pipeline_data)
                                      if app_config.is_data_source else "NA"),
                "data_check": data_check if app_config.is_data_source else True
            }
            self.logger.info(f"Payload to secondary - {response}")
        except Exception as e:
            self.logger.exception(f"Exception occurred while fetching primary device details - {e}.")
        self.logger.trace("Exiting from fetch_device_details")
        return response

    def response_action(self, data_quality):
        try:
            self.logger.trace("Entered in response_action")
            result = {
                "date_time": self.read_current_time(),
                "pipeline_version": "NA",
                "acquisition_status": False
            }
            if app_config.is_data_source:
                pipeline_data = common_utilities.read_configs(constants.channel_pipeline_path)
                result["pipeline_version"] = self.fetch_pipeline_details(pipeline_data)
                result["acquisition_status"] = (service_operations.check_ilens_client_status(logger) and not data_quality)
            return result
        except Exception as e:
            self.logger.exception(f"Exception occurred while getting response action details - {e}.")
        self.logger.trace("Exiting from response_action")
        return None
    