import base64
import json
import socket
from datetime import datetime

import httpx

from scripts.constants.app_config import app_config
from scripts.constants.app_constants import constants, services
from scripts.constants.app_variables import Counter, DataPool, DeviceInfo
from scripts.constants.events import events_constants
from scripts.utilities.common_util import common_utilities
from scripts.utilities.service_util import service_operations


def post_data(json_data, logger, endpoint="receive_data"):
    try:
        logger.trace("Entered in post_data")
        monitoring_endpoint = f"http://{app_config.host_ip}:{app_config.port_no}/{endpoint}"
        headers = {'Content-Type': 'application/json'}
        response = httpx.post(monitoring_endpoint, data=json.dumps(json_data), headers=headers, timeout=60)
        if response.status_code == 200:
            response_data = response.content.decode()
            logger.trace(f"response_data: {response_data}")
            if endpoint == "fetch_device_details":
                DataPool.data_pool = []
                response_json = (json.loads(response_data)).get("action", {})
                if (response_json.get("pipeline_version") != json_data.get("pipeline_version")
                        and response_json.get("pipeline_version") == "upgrade"):
                    file_name = constants.channel_pipeline_path
                    file_data = common_utilities.read_configs(file_name)
                    response_json = {
                        "file_name": file_name,
                        "data": file_data
                    }
                    post_data(response_json, logger)
                    post_events(event_code=events_constants.pipeline_mismatch, logger=logger)
                if response_json.get("acquisition_status"):
                    Counter.stop_counter += 1
                    if (app_config.is_data_source and Counter.stop_counter >= app_config.acquisition_restart_frequency
                            and service_operations.check_ilens_client_status(logger)):
                        if service_operations.stop_service(services.acquisition_engine, logger):
                            post_events(event_code=events_constants.primary_acq_stopped, logger=logger)
                        else:
                            post_events(event_code=events_constants.primary_acq_stopping_failed, logger=logger)
                else:
                    Counter.stop_counter = 0
    except Exception as e:
        logger.exception(f"# Error sending JSON data: {str(e)}")
    logger.trace("Exiting from post_data")


def post_events(event_code, logger, module_name=None):
    try:
        logger.trace("Entered in post_events")
        headers = {'Content-Type': 'application/json'}
        event_json = events_constants.payload_template
        event_json["timestamp"] = int((datetime.now().timestamp()) * 1000)
        event_json["ilens_device_id"] = DeviceInfo.device_id
        event_json["project_id"] = DeviceInfo.project_id
        event_json.update(events_constants.get_event_description(event_code=event_code))
        if module_name:
            event_json["event_description"] = f"{event_json['event_description']}{module_name}."
        if app_config.event_uploader_type == "udp":
            logger.trace(f"Sending events using UDP.")
            sender_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            final_message = {"topic": constants.event_topic, "data": event_json}
            bytes_to_send = base64.b64encode(json.dumps(final_message).encode())
            sender_socket.sendto(
                bytes_to_send,
                (app_config.event_uploader_host_ip, int(app_config.event_uploader_port))
            )
            sender_socket.close()
        elif app_config.event_uploader_type == "http":
            logger.trace(f"Sending events using http.")
            event_endpoint = f"{app_config.event_uploader_host_ip}{constants.event_topic}"
            response = httpx.post(
                event_endpoint,
                data=json.dumps(event_json),
                headers=headers,
                timeout=10,
                verify=False
            )
            if response.status_code == 200:
                logger.info(response.content.decode())
    except Exception as e:
        logger.exception(f"# Error sending Events: {str(e)}")
    logger.trace(f"Exiting from post_events.")
