import json
import os
import socket
from configparser import ConfigParser

import jwt
import ping3
import psutil
import yaml

from scripts.logging.logger import logger


class CommonUtilities:
    @staticmethod
    def read_configs(file_path):
        file_data = None
        logger.trace(f"Entered in read_configs with file_path: {file_path}")
        try:
            if os.path.exists(file_path):
                with open(file_path, 'r') as file:
                    if file_path.endswith("channel.yml"):
                        file_data = json.load(file)
                    elif file_path.endswith(".yml") or file_path.endswith(".yaml"):
                        file_data = yaml.safe_load(file)
                    elif file_path.endswith(".conf"):
                        file_data = file.read()
            else:
                logger.error(f"No such file - {file_path}.")
        except Exception as e:
            logger.exception(f"Error while reading configuration - {e}.")
        logger.trace("Exiting from read_configs")
        return file_data

    @staticmethod
    def ping_host(host):
        status = False
        logger.trace("Entered in the ping_host ")
        try:
            logger.trace(f"host - {host}")
            response = ping3.ping(host, timeout=2)
            if response:
                logger.info(f"Host {host} is reachable (Round-trip time: {response} ms)")
                status = True
            else:
                logger.error(f"Host {host} is not reachable")
        except socket.error as e:
            logger.exception(f"Error while trying to reach {host}: {e}")
        logger.trace("Exiting from ping_host")
        return status

    @staticmethod
    def list_the_files(files_list):
        logger.trace("Entered in list_the_files")
        try:
            return files_list.split(",")
        except Exception as e:
            logger.exception(f"Exception occurred while listing the files - {e}.")
        logger.trace("Exiting from list_the_files")
        return None

    @staticmethod
    def update_file(file_path, details):
        logger.trace("Entered in update_file")
        try:
            with open(file_path, 'w') as file:
                file.write(details)
        except Exception as e:
            logger.exception(f" - {e}.")
        logger.trace("Exiting from update_file")

    @staticmethod
    def jwt_decode(data, secret_key):
        logger.trace("Entering the jwt_decode")
        return jwt.decode(data, secret_key, algorithms=["HS256"])

    def fetch_device_details_from_lic(self, license_file_path, secret_key):
        logger.trace("Entering in fetch_device_details_from_lic")
        device_id, project_id = None, None
        try:
            if os.path.isfile(license_file_path):
                with open(license_file_path, 'r') as file:
                    encoded_data = file.read()
                decoded_data = self.jwt_decode(encoded_data, secret_key)
                device_id, project_id = decoded_data.get('ilens_device_id', ''), decoded_data.get('project_id', '')
        except Exception as e:
            logger.exception(f"Exception occurred while fetching device details - {e}.")
        logger.info(f"ilens_device_id - {device_id}, project_id - {project_id}")
        logger.trace("Exiting from fetch_device_details_from_lic")
        return device_id, project_id

    @staticmethod
    def fetch_data_from_conf(conf_file_path):
        logger.trace("Entering in fetch_data_from_conf")
        device_id, project_id = None, None
        try:
            config = ConfigParser()
            config.read(conf_file_path)
            device_id = config.get("AGENT", "agent_id")
            project_id = config.get("AGENT", "registration_project_id")
        except Exception as e:
            logger.exception(f"Exception occurred while fetching device details - {e}.")
        logger.info(f"ilens_device_id - {device_id}, project_id - {project_id}")
        logger.trace(f"ilens_device_id - {device_id}, project_id - {project_id}")
        return device_id, project_id


class WindowsUtilities:
    def find_ilens_folder(self):
        logger.trace("Entering in find_ilens_folder")
        try:
            for drive_letter in self.get_windows_drives():
                folder_path = os.path.join(drive_letter, "iLens")
                if os.path.exists(folder_path) and os.path.isdir(folder_path):
                    logger.trace("Exiting from find_ilens_folder")
                    return folder_path
        except Exception as e:
            logger.exception(f"{e}.")
        logger.trace("Exiting from find_ilens_folder")
        return None

    @staticmethod
    def get_windows_drives():
        logger.trace("Entering in get_windows_drives")
        try:
            drives = []
            for partition in psutil.disk_partitions(all=True):
                if "cdrom" not in partition.opts and partition.fstype != "":
                    drives.append(partition.device)
            logger.trace("Exiting from get_windows_drives")
            return drives
        except Exception as e:
            logger.exception(f"Exception occurred while getting windows drives - {e}.")
        logger.trace("Exiting from get_windows_drives")
        return None

    def modify_file_path(self, file_path):
        logger.trace("Entering in modify_file_path")
        modified_path = None
        try:
            drive = (self.find_ilens_folder()).replace("iLens", "")
            ilens_path = [directory for directory in ((file_path.split(":"))[1]).split("\\") if directory]
            ilens_path.insert(0, drive)
            modified_path = os.path.join(*ilens_path)
        except Exception as e:
            logger.exception(f"Exception occurred while altering the file path - {e}.")
        logger.trace("Exiting from modify_file_path")
        return modified_path


common_utilities = CommonUtilities()
